<?php

namespace App\Commands;

use CodeIgniter\CLI\CLI;
use CodeIgniter\CLI\BaseCommand;
use App\Models\NotificationEmailModel;
use Config\Services;

class NotificationsSend extends BaseCommand
{
    protected $group       = 'App';
    protected $name        = 'notifications:send';
    protected $description = 'Send scheduled notification emails whose status is pending.';
    protected $usage       = 'notifications:send';

    /**
     * Build the notification email body
     *
     * Expects $data keys:
     *  - recipient_name
     *  - type
     *  - name
     *  - timeline
     *  - notify_stage  (int days remaining)
     */
    protected function buildNotificationEmail(array $data): string
    {
        $subjectLine = ucfirst($data['type'] ?? 'Notification') . " Reminder: " . ($data['name'] ?? '');

        // Ensure stage is an int (defensive)
        $stage = isset($data['notify_stage']) ? (int) $data['notify_stage'] : null;

        // Default message if stage is missing/unexpected
        if ($stage === 1) {
            $daysMessage = "<strong style='color:red;'>This is the deadline day!</strong>";
        } elseif ($stage === 15) {
            $daysMessage = "Only <strong>15 days</strong> remaining to complete this {$data['type']}.";
        } elseif ($stage === 29) {
            $daysMessage = "Only <strong>29 days</strong> remaining to complete this {$data['type']}.";
        } else {
            // fallback
            $daysMessage = "Please check the deadline for this {$data['type']}.";
        }

        // sanitize values used inside HTML string (basic)
        $recipientName = htmlspecialchars($data['recipient_name'] ?? 'User', ENT_QUOTES, 'UTF-8');
        $itemName      = htmlspecialchars($data['name'] ?? '', ENT_QUOTES, 'UTF-8');
        $timeline      = htmlspecialchars($data['timeline'] ?? '', ENT_QUOTES, 'UTF-8');

        return "
        <div style='font-family: Arial, sans-serif; color:#333; line-height:1.6; max-width:600px; margin:auto;'>
            <h2 style='text-align:center; background:#2e86de; color:white; padding:15px; border-radius:8px 8px 0 0;'>
                {$subjectLine}
            </h2>
            <p>Hello {$recipientName},</p>
            <p>This is a reminder for your <strong>" . htmlspecialchars($data['type'] ?? '', ENT_QUOTES, 'UTF-8') . "</strong>:</p>

            <table style='width:100%; border-collapse:collapse; margin-top:15px;'>
                <tr>
                    <td style='padding:8px; border:1px solid #ccc; font-weight:bold;'>Name</td>
                    <td style='padding:8px; border:1px solid #ccc;'>{$itemName}</td>
                </tr>
                <tr>
                    <td style='padding:8px; border:1px solid #ccc; font-weight:bold;'>Deadline</td>
                    <td style='padding:8px; border:1px solid #ccc;'>{$timeline}</td>
                </tr>
                <tr>
                    <td style='padding:8px; border:1px solid #ccc; font-weight:bold;'>Status</td>
                    <td style='padding:8px; border:1px solid #ccc;'>{$daysMessage}</td>
                </tr>
            </table>

            <p style='margin-top:25px;'>Please ensure you complete all required actions before the deadline.</p>

            <div style='margin-top:30px; text-align:center; font-size:12px; color:#888;'>
                    Suite 3F4, Katu Koma Plaza, 52 Off Oladipo Diya Way, Sector Centre C, Abuja, Nigeria.<br>
                    &copy; " . date('Y') . " Learning Impact. All Rights Reserved.
                </div>
        </div>
        ";
    }

    /**
     * Execute the command
     */
    public function run(array $params)
    {
        $notificationModel = new NotificationEmailModel();

        // Fetch pending notifications (consider adding limit/pagination if many rows)
        $notifications = $notificationModel->where('status', 'pending')->findAll();

        if (empty($notifications)) {
            CLI::write('No pending notifications.', 'yellow');
            return;
        }

        foreach ($notifications as $notif) {
            $email = Services::email();
            $email->setMailType('html'); // ensure HTML

            $subject = "Reminder: {$notif['name']} ({$notif['type']})";
            $body = $this->buildNotificationEmail([
                'recipient_name' => $notif['recipient_name'] ?? 'User',
                'type'           => $notif['type'] ?? 'item',
                'name'           => $notif['name'] ?? '',
                'timeline'       => $notif['timeline'] ?? '',
                'notify_stage'   => $notif['notify_stage'] ?? null,
            ]);

            $email->setTo($notif['recipient_email']);
            $email->setSubject($subject);
            $email->setMessage($body);

            try {
                if ($email->send()) {
                    // Update status to "sent" and set sent_at timestamp
                    $notificationModel->update($notif['id'], [
                        'status'  => 'sent',
                        'sent_at' => date('Y-m-d H:i:s'),
                    ]);

                    CLI::write("Notification {$notif['id']} sent to {$notif['recipient_email']}", 'green');
                } else {
                    $error = $email->printDebugger(['headers', 'subject', 'body']);
                    log_message('error', "notifications:send - send failed for id {$notif['id']}: {$error}");
                    CLI::error("Failed to send notification {$notif['id']} to {$notif['recipient_email']}");
                }
            } catch (\Throwable $e) {
                // Ensure failed state is recorded
                $notificationModel->update($notif['id'], [
                    'status'  => 'failed',
                    'sent_at' => date('Y-m-d H:i:s'),
                ]);
                log_message('error', 'notifications:send exception - ' . $e->getMessage());
                CLI::error("Exception sending notification {$notif['id']}: " . $e->getMessage());
            }
        }
    }
}
