<?php

namespace App\Commands;

use CodeIgniter\CLI\BaseCommand;
use CodeIgniter\CLI\CLI;
use App\Libraries\EmailService;

class ProcessEmails extends BaseCommand
{
    /**
     * The Command's Group (shows under this heading in `php spark list`)
     *
     * @var string
     */
    protected $group = 'Email';

    /**
     * The Command's Name (how you invoke it)
     *
     * @var string
     */
    protected $name = 'emails:process';

    /**
     * The Command's Description
     *
     * @var string
     */
    protected $description = 'Process pending email recipients and send messages.';

    /**
     * Usage/help line
     *
     * @var string
     */
    protected $usage = 'emails:process [--limit N] [--org ID] [--dry-run]';

    /**
     * Options supported by this command
     *
     * @var array
     */
    protected $options = [
        '--limit'   => 'Maximum recipients to process this run (default 50).',
        '--org'     => 'Organization ID to scope processing (optional).',
        '--dry-run' => 'Do not send; only report what would be sent.',
    ];

    /**
     * Execute the command.
     *
     * @param array $params
     * @return void
     */
    public function run(array $params)
    {
        // Read options
        $limitOpt = CLI::getOption('limit');
        $orgOpt   = CLI::getOption('org');
        $dryRun   = CLI::getOption('dry-run') !== null;

        $limit = (int) ($limitOpt ?? 10);
        $orgId = $orgOpt !== null ? (int) $orgOpt : null;

        // Banner
        $parts = ["limit={$limit}"];
        if ($orgId) { $parts[] = "org={$orgId}"; }
        if ($dryRun) { $parts[] = "DRY RUN"; }

        CLI::write('Email processing started: ' . implode(', ', $parts), 'yellow');

        try {
            $service = new EmailService();
            $result  = $service->processPending($limit, $orgId, $dryRun);

            // Friendly output summary
            $processed = $result['processed'] ?? 0;
            $sent      = $result['sent'] ?? 0;
            $failed    = $result['failed'] ?? 0;
            $skipped   = $result['skipped'] ?? 0;
            $message   = $result['message'] ?? "Processed {$processed} | sent={$sent}, failed={$failed}, skipped={$skipped}";

            $color = $failed > 0 ? 'red' : 'green';
            CLI::write($message, $color);

        } catch (\Throwable $e) {
            CLI::error('Email processing failed: ' . $e->getMessage());
            // Log stack for later inspection
            log_message('error', 'emails:process exception - {msg} {trace}', [
                'msg'   => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }

        CLI::write('Email processing finished.', 'yellow');
    }
}
