<?php

namespace App\Controllers;

use App\Models\OrganizationsModel;
use App\Models\ThemesModel;
use App\Models\GoalsModel;
use App\Models\InitiativesModel;
use App\Models\KPIModel;
use CodeIgniter\API\ResponseTrait;
use CodeIgniter\HTTP\ResponseInterface;

class APIController extends BaseController
{
    use ResponseTrait;

    protected $organizationsModel;
    protected $themesModel;
    protected $goalsModel;
    protected $kpisModel;
    protected $initiativesModel;

    public function __construct()
    {
        $this->organizationsModel = new OrganizationsModel();
        $this->themesModel = new ThemesModel();
        $this->goalsModel = new GoalsModel();
        $this->initiativesModel = new InitiativesModel();
        $this->kpisModel = new KPIModel();
    }

    public function exportThemes($apiKey = null)
    {
        if (empty($apiKey)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Missing API key in URL.',
            ], ResponseInterface::HTTP_BAD_REQUEST);
        }

        $organization = $this->organizationsModel->where('slug', $apiKey)->first();

        if (empty($organization)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid API key provided.',
            ], ResponseInterface::HTTP_UNAUTHORIZED);
        }

        // Select only the desired columns
        $themes = $this->themesModel
            ->select('title, description, timeline, status')
            ->where('organization_id', $organization['id'])
            ->where('status', 'active')
            ->findAll();

        return $this->respond([
            'status' => 'success',
            'data' => [
                'organization' => $organization['name'],
                'themes' => $themes,
            ],
        ], ResponseInterface::HTTP_OK);
    }

    // Get organization goals
    public function exportGoals($apiKey = null)
    {
        if (empty($apiKey)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Missing API key in URL.',
            ], ResponseInterface::HTTP_BAD_REQUEST);
        }

        $organization = $this->organizationsModel->where('slug', $apiKey)->first();

        if (empty($organization)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid API key provided.',
            ], ResponseInterface::HTTP_UNAUTHORIZED);
        }

        // Fetch goals with their related theme and BSC perspective
        $goals = $this->goalsModel
            ->select('goals.description, goals.status, themes.title as theme_name, bscperspectives.name as bsc_perspective')
            ->join('themes', 'themes.id = goals.theme_id', 'left')
            ->join('bscperspectives', 'bscperspectives.id = goals.bscperspective_id', 'left')
            ->where('goals.organization_id', $organization['id'])
            ->where('goals.status', 'active')
            ->findAll();

        return $this->respond([
            'status' => 'success',
            'data' => [
                'organization' => $organization['name'],
                'goals' => $goals,
            ],
        ], ResponseInterface::HTTP_OK);
    }
    
    // export initiatives, their owners and percentage completion
public function exportInitiatives($slug)
{
    try {
        if (empty($slug)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Missing API key in URL.',
            ], ResponseInterface::HTTP_BAD_REQUEST);
        }

        $organization = $this->organizationsModel->where('slug', $slug)->first();

        if (empty($organization)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid API key provided.',
            ], ResponseInterface::HTTP_UNAUTHORIZED);
        }

        // instantiate models used to enrich each initiative
        $submissionsModel = new \App\Models\SubmissionsModel();
        $usersModel = new \App\Models\UserModel();
        $initiativeOwnersModel = new \App\Models\InitiativeOwnersModel();
        $orgStructuresModel = new \App\Models\OrganizationStructureModel();
        $goalsModel = $this->goalsModel ?? new \App\Models\GoalsModel(); // use existing or instantiate

        // fetch initiatives for the organization
        $initiatives = $this->initiativesModel
            ->where('organization_id', $organization['id'])
            ->orderBy('created_at', 'ASC')
            ->findAll();

        $now = new \DateTime();
        $result = [];

        foreach ($initiatives as $initiative) {
            $initiativeId = $initiative['id'];

            // Goal description
            $goal = $goalsModel->where('id', $initiative['goal_id'])->first();
            $goalDescription = $goal['description'] ?? null;

            // Latest submission for this initiative
            $latestSubmission = $submissionsModel
                ->where('organization_id', $organization['id'])
                ->where('project_type', 'initiative')
                ->where('initiative_id', $initiativeId)
                ->orderBy('created_at', 'DESC')
                ->first();

            // Owners
            $owners = [];
            $primaryOwnerName = null;
            $primaryOwnerType = null;

            $ownerRows = $initiativeOwnersModel
                ->where('initiative_id', $initiativeId)
                ->where('organization_id', $organization['id'])
                ->where('status', 'active')
                ->findAll();

            if (!empty($ownerRows)) {
                foreach ($ownerRows as $o) {
                    // Department / structure owner
                    if (!empty($o['organizationstructure_owner_id'])) {
                        $os = $orgStructuresModel->where('id', $o['organizationstructure_owner_id'])->first();
                        if ($os) {
                            $owners[] = [
                                'type' => $os['type'] ?? 'Structure',
                                'name' => $os['name'] ?? null,
                            ];

                            if ($primaryOwnerName === null) {
                                $primaryOwnerName = $os['name'] ?? null;
                                $primaryOwnerType = $os['type'] ?? 'Structure';
                            }
                        }
                    }

                    // Individual owner
                    if (!empty($o['individual_owner_id'])) {
                        $u = $usersModel->where('id', $o['individual_owner_id'])->first();
                        if ($u) {
                            $fullName = trim(($u['firstname'] ?? '') . ' ' . ($u['lastname'] ?? ''));
                            $owners[] = [
                                'type' => 'Individual',
                                'name' => $fullName ?: ($u['name'] ?? null),
                            ];

                            if ($primaryOwnerName === null) {
                                $primaryOwnerName = $fullName ?: ($u['name'] ?? null);
                                $primaryOwnerType = 'Individual';
                            }
                        }
                    }
                }
            }

            if (empty($owners)) {
                $owners = [];
            }

            // Compute RAG color logic
            $color = 'amber';
            $endDate = null;
            if (!empty($initiative['end_date'])) {
                try {
                    $endDate = new \DateTime($initiative['end_date']);
                } catch (\Exception $e) {
                    $endDate = null;
                }
            }

            if ($endDate && $endDate < $now) {
                $color = 'red';
            } elseif (!empty($initiative['status']) && $initiative['status'] === 'completed') {
                $color = 'green';
            } elseif (!empty($initiative['rag_status']) && $initiative['rag_status'] === 'amber') {
                $color = 'yellow';
            } elseif (!empty($initiative['current_status']) && $initiative['current_status'] === 'in_progress') {
                $color = 'yellow';
            }

            // Enrich submission with names for submitted_by and approved_by
            $submittedByName = null;
            $approvedByName = null;

            if ($latestSubmission) {
                if (!empty($latestSubmission['submitted_by'])) {
                    $submitter = $usersModel->where('id', $latestSubmission['submitted_by'])->first();
                    if ($submitter) {
                        $submittedByName = trim($submitter['name']);
                    }
                }

                if (!empty($latestSubmission['cso_id'])) {
                    $approver = $usersModel->where('id', $latestSubmission['cso_id'])->first();
                    if ($approver) {
                        $approvedByName = trim($approver['name']);
                    }
                }
            }

            // Build initiative payload
            $payload = [
                'name' => $initiative['name'],
                'target' => $initiative['target'],
                'start_date' => $initiative['start_date'],
                'end_date' => $initiative['end_date'],
                'measure_of_success' => $initiative['measure_of_success'] ?? null,
                'status' => $initiative['status'] ?? null,
                'rag_status' => $color,
                'goal' => $goalDescription,
                'owners' => $owners,
                'primary_owner_name' => $primaryOwnerName,
                'primary_owner_type' => $primaryOwnerType,
                'latest_submission' => $latestSubmission ? [
                    'submitted_comment' => trim(strip_tags($latestSubmission['submitted_comment'] ?? null)),
                    'cso_comment' => $latestSubmission['cso_comment'] ?? null,
                    'submission_status' => $latestSubmission['status'] ?? null,
                    'percentage_of_completion' => $latestSubmission['percentage_of_completion'] ?? null,
                    'approved_percentage_of_completion' => $latestSubmission['approved_percentage_of_completion'] ?? null,
                    'approved_at' => $latestSubmission['approved_at'] ?? null,
                    'submitted_by' => $submittedByName,
                    'approved_by' => $approvedByName,
                    'submitted_at' => isset($latestSubmission['created_at'])
                                    ? date('Y-m-d', strtotime($latestSubmission['created_at']))
                                    : null,

                ] : null,
            ];

            $result[] = $payload;
        }

        return $this->respond([
            'status' => 'success',
            'data' => [
                'organization' => $organization['name'],
                'initiatives' => $result,
            ],
        ], ResponseInterface::HTTP_OK);

    } catch (\Exception $e) {
        return $this->respond([
            'status' => 'error',
            'message' => 'Server error: ' . $e->getMessage(),
        ], ResponseInterface::HTTP_INTERNAL_SERVER_ERROR);
    }
}

// export KPIs with their initiative, owners, and reports
public function exportKPIs($slug)
{
    try {
        if (empty($slug)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Missing API key in URL.',
            ], ResponseInterface::HTTP_BAD_REQUEST);
        }

        $organization = $this->organizationsModel->where('slug', $slug)->first();

        if (empty($organization)) {
            return $this->respond([
                'status' => 'error',
                'message' => 'Invalid API key provided.',
            ], ResponseInterface::HTTP_UNAUTHORIZED);
        }

        // Instantiate required models
        $submissionsModel = new \App\Models\SubmissionsModel();
        $usersModel = new \App\Models\UserModel();
        $kpiOwnersModel = new \App\Models\KPIOwnersModel();
        $orgStructuresModel = new \App\Models\OrganizationStructureModel();
        $initiativesModel = new \App\Models\InitiativesModel();

        // Fetch KPIs for the organization
        $kpis = $this->kpisModel
            ->where('organization_id', $organization['id'])
            ->orderBy('created_at', 'ASC')
            ->findAll();

        $result = [];

        foreach ($kpis as $kpi) {
            $kpiId = $kpi['id'];

            // Initiative the KPI belongs to
            $initiative = $initiativesModel->where('id', $kpi['initiative_id'])->first();
            $initiativeName = $initiative['name'] ?? null;

            // Latest submission for this KPI (report)
            $latestReport = $submissionsModel
                ->where('organization_id', $organization['id'])
                ->where('project_type', 'kpi')
                ->where('kpi_id', $kpiId)
                ->orderBy('created_at', 'DESC')
                ->first();

            // Owners (department or individual)
            $owners = [];
            $primaryOwnerName = null;
            $primaryOwnerType = null;

            $ownerRows = $kpiOwnersModel
                ->where('kpi_id', $kpiId)
                ->where('organization_id', $organization['id'])
                ->where('status', 'active')
                ->findAll();

            if (!empty($ownerRows)) {
                foreach ($ownerRows as $o) {
                    // Structure owner
                    if (!empty($o['organizationstructure_owner_id'])) {
                        $os = $orgStructuresModel->where('id', $o['organizationstructure_owner_id'])->first();
                        if ($os) {
                            $owners[] = [
                                'type' => $os['type'] ?? 'Structure',
                                'name' => $os['name'] ?? null,
                            ];

                            if ($primaryOwnerName === null) {
                                $primaryOwnerName = $os['name'];
                                $primaryOwnerType = $os['type'] ?? 'Structure';
                            }
                        }
                    }

                    // Individual owner
                    if (!empty($o['individual_owner_id'])) {
                        $u = $usersModel->where('id', $o['individual_owner_id'])->first();
                        if ($u) {
                            $fullName = trim(($u['firstname'] ?? '') . ' ' . ($u['lastname'] ?? ''));
                            $owners[] = [
                                'type' => 'Individual',
                                'name' => $fullName ?: ($u['name'] ?? null),
                            ];

                            if ($primaryOwnerName === null) {
                                $primaryOwnerName = $fullName ?: ($u['name'] ?? null);
                                $primaryOwnerType = 'Individual';
                            }
                        }
                    }
                }
            }

            // Get readable submitter and approver names
            $submittedByName = null;
            $approvedByName = null;

            if ($latestReport) {
                if (!empty($latestReport['submitted_by'])) {
                    $submitter = $usersModel->where('id', $latestReport['submitted_by'])->first();
                    if ($submitter) {
                        $submittedByName = trim($submitter['name']);
                    }
                }

                if (!empty($latestReport['cso_id'])) {
                    $approver = $usersModel->where('id', $latestReport['cso_id'])->first();
                    if ($approver) {
                        $approvedByName = trim($approver['name']);
                    }
                }
            }

            // Build KPI payload
            $payload = [
                'name' => $kpi['name'],
                'timeline' => $kpi['timeline'] ?? null,
                'status' => $kpi['status'] ?? null,
                'initiative' => $initiativeName,
                'owners' => $owners,
                'primary_owner_name' => $primaryOwnerName,
                'primary_owner_type' => $primaryOwnerType,
                'latest_report' => $latestReport ? [
                    'submitted_comment' => trim(strip_tags($latestReport['submitted_comment'] ?? '')),
                    'cso_comment' => trim(strip_tags($latestReport['cso_comment'] ?? '')),
                    'report_status' => $latestReport['status'] ?? null,
                    'percentage_of_completion' => $latestReport['percentage_of_completion'] ?? null,
                    'approved_percentage_of_completion' => $latestReport['approved_percentage_of_completion'] ?? null,
                    'approved_at' => $latestReport['approved_at'] ?? null,
                    'submitted_by' => $submittedByName,
                    'approved_by' => $approvedByName,
                    'submitted_at' => isset($latestReport['created_at'])
                        ? date('Y-m-d', strtotime($latestReport['created_at']))
                        : null,
                ] : null,
            ];

            $result[] = $payload;
        }

        return $this->respond([
            'status' => 'success',
            'data' => [
                'organization' => $organization['name'],
                'kpis' => $result,
            ],
        ], ResponseInterface::HTTP_OK);

    } catch (\Exception $e) {
        return $this->respond([
            'status' => 'error',
            'message' => 'Server error: ' . $e->getMessage(),
        ], ResponseInterface::HTTP_INTERNAL_SERVER_ERROR);
    }
}


}
