<?php

namespace App\Controllers;

use Google\Client;
use Google\Service\Calendar;
use Google\Service\Exception as GoogleServiceException;
use App\Models\KPIModel;
use App\Models\AuthTokensModel;
use App\Models\InitiativesModel;
use App\Controllers\BaseController;

class CalendarController extends BaseController
{
    protected $initiativesModel;
    protected $kpisModel;
    protected $authTokensModel;

    public function __construct()
    {
        $this->initiativesModel = new InitiativesModel();
        $this->kpisModel = new KPIModel();
        $this->authTokensModel = new AuthTokensModel();
    }

    private function getClient(): Client
    {
        $client = new Client();
        $client->setClientId(getenv('google.client_id'));
        $client->setClientSecret(getenv('google.client_secret'));
        $client->setRedirectUri(base_url('google-calendar/callback'));
        $client->addScope(Calendar::CALENDAR);
        $client->setAccessType('offline');   // ensures refresh_token is returned
        $client->setPrompt('consent');       // ensures refresh_token on first consent

        return $client;
    }

    public function googleRender()
    {
        $type = $this->request->getGet('type');
        $slug = $this->request->getGet($type);

        // Fetch record
        if ($type === 'initiative') {
            $record = $this->initiativesModel->where('slug', $slug)->first();
        } elseif ($type === 'kpi') {
            $record = $this->kpisModel->where('slug', $slug)->first();
        } else {
            return redirect()->back()->with('error', 'Invalid type');
        }

        if (!$record) {
            return redirect()->back()->with('error', ucfirst($type) . ' not found.');
        }

        // Find stored token
        $tokenRow = $this->authTokensModel
            ->where('user_id', session()->get('user_id'))
            ->first();

        if (!$tokenRow) {
            return redirect()->to(base_url("google-calendar/consent?type=$type&slug=$slug"));
        }

        $client = $this->getClient();
        $storedToken = json_decode($tokenRow['access_token'], true);
        $client->setAccessToken($storedToken);

        // Refresh token if needed
        if ($client->isAccessTokenExpired()) {
            if (!empty($tokenRow['refresh_token'])) {
                try {
                    $newToken = $client->fetchAccessTokenWithRefreshToken($tokenRow['refresh_token']);

                    if (isset($newToken['error']) && $newToken['error'] === 'invalid_grant') {
                        return redirect()->to(base_url("google-calendar/consent?type=$type&slug=$slug"))
                                         ->with('error', 'Google authorization expired, please reconnect.');
                    }

                    // Merge old refresh_token if Google didn’t return one
                    if (!isset($newToken['refresh_token'])) {
                        $newToken['refresh_token'] = $tokenRow['refresh_token'];
                    }

                    $this->authTokensModel->update($tokenRow['id'], [
                        'access_token'  => json_encode($client->getAccessToken()),
                        'refresh_token' => $newToken['refresh_token'],
                    ]);
                } catch (\Exception $e) {
                    return redirect()->to(base_url("google-calendar/consent?type=$type&slug=$slug"))
                                     ->with('error', 'Google authorization failed, please reconnect.');
                }
            } else {
                return redirect()->to(base_url("google-calendar/consent?type=$type&slug=$slug"))
                                 ->with('error', 'Missing Google refresh token, please reconnect.');
            }
        }

        $service = new Calendar($client);

        // Dates
        if (!empty($record['end_date'])) {
            $startDate = date('c', strtotime($record['end_date']));
            $endDate   = date('c', strtotime($record['end_date'] . ' +1 hour'));
        } elseif (!empty($record['timeline'])) {
            $startDate = date('c', strtotime($record['timeline']));
            $endDate   = date('c', strtotime($record['timeline'] . ' +1 hour'));
        } else {
            $startDate = date('c');
            $endDate   = date('c', strtotime('+1 hour'));
        }

        $event = new \Google\Service\Calendar\Event([
            'summary'     => $record['name'] ?? $record['kpi_statement'],
            'description' => $record['brief'] ?? $record['description'] ?? '',
            'start'       => ['dateTime' => $startDate],
            'end'         => ['dateTime' => $endDate],
        ]);

        try {
            $service->events->insert('primary', $event);
        } catch (GoogleServiceException $e) {
            if ($e->getErrors()[0]['reason'] === 'invalid_grant') {
                return redirect()->to(base_url("google-calendar/consent?type=$type&slug=$slug"))
                                 ->with('error', 'Google authorization revoked, please reconnect.');
            }
            throw $e; // let other errors bubble up
        }

        return redirect()->back()->with('success', ucfirst($type) . ' added to Google Calendar');
    }

    public function googleConsent()
    {
        $type = $this->request->getGet('type');
        $slug = $this->request->getGet('slug');

        $client = $this->getClient();
        $authUrl = $client->createAuthUrl();

        session()->set([
            'calendar_type' => $type,
            'calendar_slug' => $slug,
        ]);

        return redirect()->to($authUrl);
    }

    public function googleCallback()
    {
        $client = $this->getClient();

        if ($this->request->getGet('code')) {
            $token = $client->fetchAccessTokenWithAuthCode($this->request->getGet('code'));

            if (isset($token['error'])) {
                return redirect()->to('/')
                                 ->with('error', 'Authorization failed: ' . ($token['error_description'] ?? $token['error']));
            }

            // Upsert token in DB
            $existing = $this->authTokensModel->where('user_id', session()->get('user_id'))->first();
            $data = [
                'user_id'       => session()->get('user_id'),
                'access_token'  => json_encode($client->getAccessToken()),
                'refresh_token' => $token['refresh_token'] ?? ($existing['refresh_token'] ?? null),
            ];

            if ($existing) {
                $this->authTokensModel->update($existing['id'], $data);
            } else {
                $this->authTokensModel->insert($data);
            }

            $type = session()->get('calendar_type');
            $slug = session()->get('calendar_slug');

            return redirect()->to(base_url("google-calendar/render?type=$type&$type=$slug"));
        }

        return redirect()->to('/')->with('error', 'Authorization failed.');
    }
}
