<?php

namespace App\Controllers;

use App\Models\ChatsModel;
use App\Models\ChatMessagesModel;
use App\Models\OrganizationsModel;
use App\Controllers\BaseController;
use CodeIgniter\HTTP\ResponseInterface;

class ChatsController extends BaseController
{
    protected $organizationsModel;
    protected $organization;
    protected $chatsModel;
    protected $chatMessagesModel;

    public function __construct()
    {
        $this->organizationsModel = new OrganizationsModel();
        $this->organization = $this->organizationsModel->where('slug', session()->get('slug'))->first();
        $this->chatsModel = new ChatsModel();
        $this->chatMessagesModel = new ChatMessagesModel();
    }

    // select organization's chats with its initiator
  public function index()
{
    $status = $this->request->getGet('status'); 

    $builder = $this->chatsModel
        ->select('chats.*, users.name AS initiator_name, users.email AS initiator_email')
        ->join('users', 'users.id = chats.initiated_by')
        ->where('chats.organization_id', $this->organization['id'])
        ->orderBy('chats.created_at', 'DESC');

    if (!empty($status)) {
        $builder->where('chats.status', $status);
    }

    $chats = $builder->findAll();

    return view('chats/index', [
        'organization' => $this->organization,
        'chats' => $chats
    ]);
}


    // store new chat in db
    public function store()
    {
        $rules = [
            'title' => 'required|max_length[255]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->with('error', 'Chat title empty or too long!');
        }

        $data = [
            'slug' => substr(md5(uniqid(time(), true)), 0, 32),
            'organization_id' => $this->organization['id'],
            'title' => $this->request->getPost('title'),
            'initiated_by' => session()->get('user_id'),
        ];

        if ($this->chatsModel->save($data)) {
            return redirect()->back()->with('success', 'Chat initiated successfully');
        }
    }

// view a single conversation
public function view()
{
    $slug = $this->request->getGet('chat');

    // First, get the chat record by slug
    $chat = $this->chatsModel
        ->where('slug', $slug)
        ->first();

    if (!$chat) {
        return redirect()->back()->with('error', 'Chat not found!');
    }

    // Fetch chat messages with sender info and reply-to info
    $builder = $this->chatMessagesModel
        ->select('
            chatmessages.*,
            users.name AS sender_name,
            chats.slug AS chat_slug,
            chats.title AS chat_title,
            reply.id AS reply_id,
            reply.message AS reply_message,
            reply_user.name AS reply_sender_name
        ')
        ->join('users', 'users.id = chatmessages.user_id')
        ->join('chats', 'chats.id = chatmessages.chat_id')
        ->join('chatmessages AS reply', 'reply.id = chatmessages.reply_to_message_id', 'left')
        ->join('users AS reply_user', 'reply_user.id = reply.user_id', 'left')
        ->where('chatmessages.chat_id', $chat['id'])
        ->orderBy('chatmessages.created_at', 'ASC');

    $messages = $builder->findAll();

    return view('chats/view', [
        'chat' => $chat,
        'messages' => $messages,
        'organization' => $this->organization,
    ]);
}

    // store a reply/message to a chat
    public function storeChatReply()
    {
        $rules = [
            'message' => 'required|max_length[2000]',
        ];

        if(!$this->validate($rules))
        {
            return redirect()->back()->with('error', 'Chat reply is empty or too long');
        }

        $data = [
            'slug' => substr(md5(uniqid(time(), true)), 0, 32),
            'organization_id' => $this->organization['id'],
            'chat_id' => $this->request->getPost('id'),
            'user_id' => session()->get('user_id'),
            'message' => $this->request->getPost('message'),
        ];

        if($this->chatMessagesModel->save($data))
        {
            return redirect()->back()->with('success', 'Message saved successfully');
        }
    }
    
      // store a reply/message to a chat
    public function storeMessageReply()
    {
        $rules = [
            'message_reply' => 'required|max_length[2000]',
        ];

        if(!$this->validate($rules))
        {
            return redirect()->back()->with('error', 'Message reply is empty or too long');
        }

        $data = [
            'slug' => substr(md5(uniqid(time(), true)), 0, 32),
            'organization_id' => $this->organization['id'],
            'chat_id' => $this->request->getPost('chat_id'),
            'user_id' => session()->get('user_id'),
            'message' => $this->request->getPost('message_reply'),
            'reply_to_message_id' => $this->request->getPost('message_id'),
        ];
        
        // dd($data);

        if($this->chatMessagesModel->save($data))
        {
            return redirect()->back()->with('success', 'Message reply saved successfully');
        }
    }

    // disable a chat by cso
    public function disable($id)
    {
        $data = [
            'status' => 'inactive',
        ];

        if($this->chatsModel->update($id, $data))
        {
            return redirect()->back()->with('success', 'Chat disabled successfully');
        }
    }

        // enable a chat by cso
    public function enable($id)
    {
        $data = [
            'status' => 'active',
        ];

        if($this->chatsModel->update($id, $data))
        {
            return redirect()->back()->with('success', 'Chat enabled successfully');
        }
    }
}
