<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\CustomizationModel;
use App\Models\OrganizationsModel;
use CodeIgniter\HTTP\ResponseInterface;

class CustomizationsController extends BaseController
{
    protected $CustomizationModel;
    protected $organizationsModel;

    public function __construct()
    {
        $this->CustomizationModel = new CustomizationModel();
        $this->organizationsModel = new OrganizationsModel();
    }

    public function edit($slug)
    {
        $organization = $this->organizationsModel->where('slug', $slug)->first();
        // dd($organization);

        return view('customization/edit', ['organization' => $organization]);
    }

    // create new organization details
    public function store()
    {
        $rules = [
            'organization_name' => 'required|string',
            'logo' => 'is_image[logo]|mime_in[logo,image/jpg,image/jpeg,image/png]|max_size[logo,5120]'
        ];

        // If logo is not uploaded, remove 'uploaded' rule
        // if (!$this->request->getFile('logo')->isValid()) {
        //     $rules['logo'] = 'permit_empty|is_image[logo]|mime_in[logo,image/jpg,image/jpeg,image/png]|max_size[logo,5120]';
        // }

        if (!$this->validate($rules)) {
            return view('customization/create', [
                'validation' => $this->validator
            ]);
        }

        $orgName = $this->request->getPost('organization_name');
        $orgSlug = url_title($orgName, '_', true); // to generate safe filename

        $logo = $this->request->getFile('logo');
        $logoPath = null;

        if ($logo && $logo->isValid() && !$logo->hasMoved()) {
            $newName = $logo->getRandomName();
            $uploadPath = FCPATH . 'uploads/organizations/';

            // Make sure the directory exists or create new one
            if (!is_dir($uploadPath)) {
                mkdir($uploadPath, 0777, true);
            }

            $logo->move($uploadPath, $newName);
            $logoPath = 'uploads/organizations/' . $newName;
        }


        $this->CustomizationModel->save([
            'organization_name' => $orgName,
            'logo' => $logoPath
        ]);

        return redirect()->to('organization/create')->with('success', 'Customization settings saved successfully.');
    }

    // update organization details
    public function update($slug)
    {


        $organization = $this->organizationsModel->where('slug', $slug)->first();

        if (!$organization) {
            return redirect()->back()->with('error', 'Organization not found.');
        }

        $id = $this->request->getPost('id');
        $organization = $this->organizationsModel->find($id);

        $validationRules = [
            'name' => 'required|max_length[255]|string',
            'email' => "required|valid_email|is_unique[organizations.email,id,{$id}]",
            'logo' => 'is_image[logo]|mime_in[logo,image/jpg,image/jpeg,image/png]|max_size[logo,5120]'
        ];

        $validationMessages = [
            'name' => [
                'required' => 'The name field is required',
                'string'   => 'The name must be a valid string',
            ],
            'email' => [
                'required'    => 'The email field is required.',
                'valid_email' => 'Please provide a valid email address',
                'is_unique'   => 'This email has already been taken.'
            ],
        ];

        if (!$this->validate($validationRules, $validationMessages)) {
            return view('customization/edit', [
                'validation' => $this->validator,
                'organization' => $organization
            ]);
        }

        $orgName = $this->request->getPost('name');
        $orgSlug = url_title($orgName, '_', true);
        $logo = $this->request->getFile('logo');

        $logoPath = $organization['logo']; // default to existing logo

        // If a new logo was uploaded
        if ($logo && $logo->isValid() && !$logo->hasMoved()) {
            $uploadPath = FCPATH . 'uploads/organizations/';
            $newName = $logo->getRandomName();

            if (!is_dir($uploadPath)) {
                mkdir($uploadPath, 0777, true);
            }

            // Delete old logo file if it exists
            if (!empty($organization['logo'])) {
                $oldLogoPath = FCPATH . $organization['logo'];
                if (file_exists($oldLogoPath)) {
                    unlink($oldLogoPath);
                }
            }

            // Move new file and update logoPath
            $logo->move($uploadPath, $newName);
            $logoPath = 'uploads/organizations/' . $newName;
        }
        
        $data = [
            'name'  => $orgName,
            'email' => $this->request->getPost('email'),
            'logo'  => $logoPath
        ];
        
        // Check if the 'api_key' (checkbox) was checked in the form
        if ($this->request->getPost('api_key')) {
            // Generate a new random slug (32 characters)
            $data['slug'] = substr(md5(uniqid((string) time(), true)), 0, 32);
        
            // Update the session slug to the new value
            session()->set('slug', $data['slug']);
            $slug = $data['slug'];
        }
        
        $this->organizationsModel->update($id, $data);

        return redirect()->to('organization/update/' . $slug)->with('success', 'Details updated successfully.');
    }

    // return dashboard with organization's details (dashboard) for super admin
    public function home($slug)
    {
        $organization = $this->organizationsModel->where('slug', $slug)->first();
        session()->set('slug', $organization['slug']);

        return redirect()->to('home');
    }

    //display form to update stracker logo and address
    public function strackerSettings()
    {
        $stracker = $this->CustomizationModel->first();
        // dd($stracker);

        return view('customization/stracker', ['stracker' => $stracker]);
    }

    // store stracker settings
    public function storeStrackerSettings()
    {
        $rules = [
            'lim_address' => 'required|string',
            'logo' => 'is_image[logo]|mime_in[logo,image/jpg,image/jpeg,image/png]|max_size[logo,5120]'
        ];

        if (!$this->validate($rules)) {
            return view('customization/stracker', [
                'validation' => $this->validator
            ]);
        }

        // Check if a logo is uploaded
        $file = $this->request->getFile('logo');
        $filePresent = $file && $file->isValid() && !$file->hasMoved();

        $fileLink = null;

        if ($filePresent) {
            $uploadPath = FCPATH . 'uploads/lim/';
            $newName = 'stracker_logo_' . $file->getRandomName();

            // Create the folder if it doesn't exist
            if (!is_dir($uploadPath)) {
                mkdir($uploadPath, 0777, true);
            }

            // Move the file to public/uploads/submissions
            $file->move($uploadPath, $newName);

            $fileLink = 'uploads/lim/' . $newName;
        }


        $this->CustomizationModel->save([
            'lim_address' => $this->request->getPost('lim_address'),
            'logo' => $fileLink
        ]);

        return redirect()->to('settings')->with('success', 'LIM settings saved successfully');
    }

    // update stracker settings
    public function updateStrackerSettings()
    {
        $rules = [
            'lim_address' => 'required|string',
            'logo' => 'is_image[logo]|mime_in[logo,image/jpg,image/jpeg,image/png]|max_size[logo,5120]'
        ];

        if (!$this->validate($rules)) {
            return view('customization/stracker', [
                'validation' => $this->validator
            ]);
        }

        // Fetch existing settings (assumes only one settings row exists)
        $existing = $this->CustomizationModel->first();

        $file = $this->request->getFile('logo');
        $filePresent = $file && $file->isValid() && !$file->hasMoved();

        $fileLink = $existing['logo']; // default to existing logo path

        if ($filePresent) {
            $uploadPath = FCPATH . 'uploads/lim/';
            $newName = 'stracker_logo_' . $file->getRandomName();

            // Create the folder if it doesn't exist
            if (!is_dir($uploadPath)) {
                mkdir($uploadPath, 0777, true);
            }

            // Delete old logo file if it exists
            if (!empty($existing['logo'])) {
                $oldFilePath = FCPATH . $existing['logo'];
                if (file_exists($oldFilePath)) {
                    unlink($oldFilePath);
                }
            }

            // Move the new file
            $file->move($uploadPath, $newName);
            $fileLink = 'uploads/lim/' . $newName;
        }

        // Update record
        $this->CustomizationModel->update($existing['id'], [
            'lim_address' => $this->request->getPost('lim_address'),
            'logo' => $fileLink
        ]);

        return redirect()->to('settings')->with('success', 'LIM settings updated successfully');
    }
}
