<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\EmailRecipientsModel;
use App\Models\EmailsModel;
use App\Models\OrganizationsModel;
use App\Models\OrganizationStructureModel;
use App\Models\UserModel;
use CodeIgniter\HTTP\ResponseInterface;

class EmailController extends BaseController
{
    protected $email;
    protected $organizationsModel;
    protected $organization;
    protected $emailsModel;
    protected $emailRecipientsModel;
    protected $userModel;
    protected $structuresModel;

    public function __construct()
    {
        $this->email = \Config\Services::email();
        $this->organizationsModel = new OrganizationsModel();
        $this->organization = $this->organizationsModel->where('slug', session()->get('slug'))->first();
        $this->emailsModel = new EmailsModel();
        $this->emailRecipientsModel = new EmailRecipientsModel();
        $this->userModel = new UserModel();
        $this->structuresModel = new OrganizationStructureModel();
    }

    public function sendMail($recipient, $subject, $message)
    {

        $this->email->setTo($recipient);

        $this->email->setSubject($subject);
        $this->email->setMessage($message);

        if ($this->email->send()) {
            return true;
        } else {
            // 👇 This shows the error message and exits
            echo $this->email->printDebugger(['headers', 'subject', 'body']);
            exit;
        }
    }

    // display all emails
    public function index()
    {
        $emails = $this->emailsModel
            ->select('emails.*, users.name as sender_name')
            ->join('users', 'users.id = emails.sent_by', 'left')
            ->where('emails.organization_id', $this->organization['id'])
            ->orderBy('emails.created_at', 'DESC')
            ->findAll();

        return view('emails/index', [
            'emails' => $emails,
            'organization' => $this->organization,
        ]);
    }


    // display form for cso to send emails
    public function create()
    {
        $users = $this->userModel->where('organizations_id', $this->organization['id'])->findAll();
        $structures = $this->structuresModel->where('organization_id', $this->organization['id'])->findAll();

        return view('emails/create', ['users' => $users, 'structures' => $structures]);
    }

    // collect email details and recipients to store in db
    public function store()
    {
        $rules = [
            'recipient_type' => [
                'rules' => 'required|in_list[individual,group]',
                'errors' => [
                    'required' => 'Please select a recipient type',
                    'in_list' => 'Invalid recipient type selected',
                ],
            ],
            'recipients' => [
                'rules' => 'required',
                'errors' => [
                    'required' => 'Please select at least one recipient',
                ],
            ],
            'subject' => [
                'rules' => 'required|max_length[255]',
                'errors' => [
                    'required' => 'The email subject is required',
                    'max_length' => 'The subject is too long',
                ],
            ],
            'body' => [
                'rules' => 'required',
                'errors' => [
                    'required' => 'The email body is required',
                ],
            ],
        ];

        if (!$this->validate($rules)) {
            // Reload form with validation errors
            $users = $this->userModel
                ->where('organizations_id', $this->organization['id'])
                ->where('status', 'active')
                ->findAll();

            $structures = $this->structuresModel
                ->where('organization_id', $this->organization['id'])
                ->findAll();

            return view('emails/create', [
                'users' => $users,
                'structures' => $structures,
                'organization' => $this->organization,
                'validation' => $this->validator
            ]);
        }

        // Store the email
        $emailData = [
            'organization_id' => $this->organization['id'],
            'slug'            => substr(md5(uniqid(time(), true)), 0, 40),
            'subject'         => $this->request->getPost('subject'),
            'body'            => $this->request->getPost('body'),
            'sent_by'         => session()->get('user_id'),
            'status'          => 'pending', // not sent yet
        ];

        $this->emailsModel->insert($emailData);
        $emailId = $this->emailsModel->getInsertID();

        // Prepare recipients list
        $recipients = $this->request->getPost('recipients');
        $recipientType = $this->request->getPost('recipient_type');

        foreach ($recipients as $recipient) {
            if ($recipientType === 'individual' && strpos($recipient, 'user_') === 0) {
                $userId = (int) str_replace('user_', '', $recipient);
                $user = $this->userModel->find($userId);

                if ($user) {
                    $this->emailRecipientsModel->insert([
                        'organization_id' => $this->organization['id'],
                        'email_id'        => $emailId,
                        'recipient_id'    => $user['id'],
                        'structure_id'    => null,
                        'email_address'   => $user['email'],
                        'status'          => 'pending',
                        'sent_at'         => null
                    ]);
                }
            } elseif ($recipientType === 'group' && strpos($recipient, 'group_') === 0) {
                $structureId = (int) str_replace('group_', '', $recipient);
                $structure = $this->structuresModel->find($structureId);

                if ($structure) {
                    // Fetch all users in this structure
                    $usersInStructure = $this->userModel
                        ->where('organizationstructures_id', $structure['id'])
                        ->where('status', 'active')
                        ->findAll();

                    foreach ($usersInStructure as $u) {
                        $this->emailRecipientsModel->insert([
                            'organization_id' => $this->organization['id'],
                            'email_id'        => $emailId,
                            'recipient_id'    => $u['id'],
                            'structure_id'    => $structure['id'],
                            'email_address'   => $u['email'],
                            'status'          => 'pending',
                            'sent_at'         => null
                        ]);
                    }
                }
            }
        }

        return redirect()->to('emails')->with('success', 'Email sent successfully');
    }

    public function cron_processPendingEmails()
    {
        // dd('ok');
        // Ensure this method is run only from CLI
        // if (!is_cli()) {
        //     throw new \CodeIgniter\Exceptions\PageNotFoundException('Page not found');
        // }
        // echo "Cron started...\n";
        
        // Get pending recipients with their email details
        $pendingRecipients = $this->emailRecipientsModel
            ->select('emailrecipients.*, emails.subject, emails.body')
            ->join('emails', 'emails.id = emailrecipients.email_id')
            ->where('emailrecipients.status', 'pending')
            ->where('emails.status', 'pending')
            ->limit(5) // batch size to avoid timeouts
            ->findAll();

        if (empty($pendingRecipients)) {
            // echo "No pending emails found.\n";
            return;
        }
    //      print_r($pendingRecipients);
    // exit;

        $processedEmailIds = [];

        foreach ($pendingRecipients as $recipient) {
            $sent = $this->sendMail(
                $recipient['email_address'],
                $recipient['subject'],
                $recipient['body']
            );

            if ($sent) {
                // Update recipient record
                $this->emailRecipientsModel->update($recipient['id'], [
                    'status' => 'sent',
                    'sent_at' => date('Y-m-d H:i:s')
                ]);

                // Keep track of the email_id to later mark the parent email as fully sent
                $processedEmailIds[] = $recipient['email_id'];
            }
        }

        // After sending, check if each email has any pending recipients left
        $processedEmailIds = array_unique($processedEmailIds);

        foreach ($processedEmailIds as $emailId) {
            $pendingCount = $this->emailRecipientsModel
                ->where('email_id', $emailId)
                ->where('status', 'pending')
                ->countAllResults();

            if ($pendingCount === 0) {
                $this->emailsModel->update($emailId, ['status' => 'sent']);
            }
        }

        // echo "Pending emails processed successfully.\n";
    }

    // display the details of a signle email
    public function view()
    {
        $emailSlug = $this->request->getGet('email');

        // Fetch the email along with the sender's information
        $emailRecord = $this->emailsModel
            ->select('emails.*, sender.name AS sender_name')
            ->join('users AS sender', 'sender.id = emails.sent_by', 'left')
            ->where('emails.slug', $emailSlug)
            ->where('emails.organization_id', $this->organization['id'])
            ->first();

        if (!$emailRecord) {
            return redirect()->back()->with('error', 'Email not found.');
        }

        // Fetch all recipients for this email along with:
        // - User details if it's an individual recipient
        // - Structure details if it's sent to a group (department, SBU, etc.)
        $emailRecipients = $this->emailRecipientsModel
            ->select(
                'emailrecipients.*,
                recipientUser.name AS recipient_name,
                recipientUser.email AS recipient_email,
                recipientStructure.name AS structure_name'
            )
            ->join('users AS recipientUser', 'recipientUser.id = emailrecipients.recipient_id', 'left')
            ->join('organizationstructures AS recipientStructure', 'recipientStructure.id = emailrecipients.structure_id', 'left')
            ->where('emailrecipients.email_id', $emailRecord['id'])
            ->where('emailrecipients.organization_id', $this->organization['id'])
            ->findAll();

            // dd($emailRecord);
        // Return data to the view for rendering
        return view('emails/view', [
            'emailRecord'         => $emailRecord,
            'emailRecipients'    => $emailRecipients,
            'organization'  => $this->organization,
        ]);
    }
}
