<?php

namespace App\Controllers;

use App\Models\ThemesModel;
use App\Controllers\BaseController;
use App\Models\BSCPerspectivesModel;
use App\Models\GoalsModel;
use App\Models\OrganizationsModel;
use CodeIgniter\HTTP\ResponseInterface;

class GoalsController extends BaseController
{
    protected $themesModel;
    protected $bscperspectiveModel;
    protected $goalsModel;
    protected $themes;
    protected $bscperspectives;
    protected $organizationsModel;
    protected $organization;

    public function __construct()
    {
        $this->themesModel = new ThemesModel();
        $this->bscperspectiveModel = new BSCPerspectivesModel();
        $this->goalsModel = new GoalsModel();
        $this->organizationsModel = new OrganizationsModel();
        $this->organization = $this->organizationsModel->where('slug', session()->get('slug'))->first();
        $this->bscperspectives = $this->bscperspectiveModel->where('organizations_id', $this->organization['id'])->where('status', 'active')->get()->getResultArray();
        $this->themes = $this->themesModel->where('organization_id', $this->organization['id'])->where('status', 'active')->findAll();
    }

    public function index()
    {
        $status = $this->request->getGet('status');

        $builder = $this->goalsModel
            ->select('goals.*, themes.title as theme_name, themes.timeline, themes.status as theme_status, bscperspectives.name as bscbscperspective_name')
            ->join('themes', 'themes.id = goals.theme_id')
            ->join('bscperspectives', 'bscperspectives.id = goals.bscperspective_id', 'left')
            ->where('goals.organization_id', $this->organization['id']);

        if (!is_null($status)) {
            if ($status === 'due') {
                // Custom logic for due goals
                $builder
                    ->whereIn('goals.status', ['active', 'pending'])
                    ->where('themes.status', 'active')
                    ->where('themes.timeline <', date('Y-m-d H:i:s'));
            } else {
                $builder->where('goals.status', $status)
                    ->where('themes.status', 'active'); // ensure theme is still active for normal statuses
            }
        } else {
            // Default: show only goals under active themes
            $builder->where('themes.status', 'active');
        }

        $goals = $builder
            ->orderBy('goals.status')
            ->orderBy('goals.created_at', 'DESC')
            ->findAll();

        $action = $this->request->getGet('action');
        if ($action === 'download') {
            return view('downloads/goals', [
                'organization' => $this->organization,
                'goals' => $goals
            ]);
        }

        return view('goals/index', [
            'organization' => $this->organization,
            'goals' => $goals
        ]);
    }



    // display form to create goals
    public function create()
    {
        return view('goals/create', ['organization' => $this->organization, 'themes' => $this->themes, 'bscperspectives' => $this->bscperspectives]);
    }

    // collect user input to create goals and store in db
    public function store()
    {
        $rules = [
            'theme_id' => 'required',
            'description' => 'required|string',
        ];

        $messages = [
            'theme_id' => [
                'required' => 'Choose a theme',
            ],
            'description' => [
                'required' => 'Add the details of the goal',
            ],
        ];

        if (!$this->validate($rules, $messages)) {
            return view('goals/create', ['organization' => $this->organization, 'validation' => $this->validator, 'themes' => $this->themes, 'bscperspectives' => $this->bscperspectives]);
        }

        $data = [
            'organization_id' => $this->organization['id'],
            'theme_id' => $this->request->getPost('theme_id') ?: null,
            'bscperspective_id' => $this->request->getPost('bscperspective_id') ?: null,
            'slug' => substr(md5(uniqid(time(), true)), 0, 32),
            'description' => $this->request->getPost('description'),
        ];

        if ($this->goalsModel->save($data)) {
            return redirect()->to('goals')->with('success', 'Goal created successfully');
        }
    }

    // display form to edit goals
    public function edit($slug)
    {
        $goal = $this->goalsModel->where('slug', $slug)->first();
        // dd($goal);

        return view('goals/edit', ['organization' => $this->organization, 'goal' => $goal, 'themes' => $this->themes, 'bscperspectives' => $this->bscperspectives]);
    }

    // collect data and update goal in db
    public function update()
    {
        // dd('ok');
        $id = $this->request->getPost('id');
        $goal = $this->goalsModel->find($id);

        $rules = [
            'theme_id' => 'required',
            'description' => 'required|string',
        ];

        $messages = [
            'theme_id' => [
                'required' => 'Choose a theme',
            ],
            'description' => [
                'required' => 'Add the details of the goal',
            ],
        ];

        if (!$this->validate($rules, $messages)) {
            return view('goals/edit', ['organization' => $this->organization, 'goal' => $goal, 'validation' => $this->validator, 'themes' => $this->themes, 'bscperspectives' => $this->bscperspectives]);
        }

        $data = [
            'theme_id' => $this->request->getPost('theme_id') ?: null,
            'bscperspective_id' => $this->request->getPost('bscperspective_id') ?: null,
            'description' => $this->request->getPost('description'),
        ];

        if ($this->goalsModel->update($id, $data)) {
            return redirect()->to('goals')->with('success', 'Goal updated successfully');
        }
    }

    // disable goal
    public function disable($id)
    {
        // dd('disable');
        $data = [
            'status' => 'inactive',
        ];

        if ($this->goalsModel->update($id, $data)) {
            return redirect()->to('goals')->with('success', 'Goal disabled successfully');
        }
        return redirect()->to('goals')->with('error', 'Unable to disable goal');
    }

    // enable goal
    public function enable($id)
    {
        // dd('enable');
        $data = [
            'status' => 'active',
        ];

        if ($this->goalsModel->update($id, $data)) {
            return redirect()->to('goals')->with('success', 'Goal enabled successfully');
        }
        return redirect()->to('goals')->with('error', 'Unable to disable goal');
    }

    // mark a goal complete
    public function markComplete($id)
    {
        $data = [
            'status' => 'completed',
        ];

        if ($this->goalsModel->update($id, $data)) {
            return redirect()->to('goals')->with('success', 'Goal marked completed');
        }
    }
}
