<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\GoalsModel;
use App\Models\InitiativeOwnersModel;
use App\Models\InitiativesModel;
use App\Models\OrganizationsModel;
use App\Models\OrganizationStructureModel;
use App\Models\UserModel;
use CodeIgniter\HTTP\ResponseInterface;

class InitiativesController extends BaseController
{
    protected $goalsModel;
    protected $initiativesModel;
    protected $organizationStructureModel;
    protected $userModel;
    protected $initiativesOwnerModel;
    protected $organizationsModel;
    protected $organization;

    public function __construct()
    {
        $this->goalsModel = new GoalsModel();
        $this->initiativesModel = new InitiativesModel();
        $this->organizationStructureModel = new OrganizationStructureModel();
        $this->userModel = new UserModel();
        $this->initiativesOwnerModel = new InitiativeOwnersModel();
        $this->organizationsModel = new OrganizationsModel();
        $this->organization = $this->organizationsModel->where('slug', session()->get('slug'))->first();
    }

    public function index()
{
    $action = $this->request->getGet('action');
    $status = $this->request->getGet('status');

    // Base query with join to goals table to filter by goal status = 'active'
    $builder = $this->initiativesModel
        ->select('initiatives.*') // in case of column overlap
        ->join('goals', 'goals.id = initiatives.goal_id', 'inner')
        ->where('goals.status', 'active')
        ->where('initiatives.organization_id', $this->organization['id']);

    if ($status === 'due') {
        // Custom logic to fetch due initiatives (end_date < now and not completed)
        $builder->where('initiatives.status !=', 'completed')
                ->where('initiatives.status !=', 'inactive')
                ->where('initiatives.end_date <', date('Y-m-d H:i:s'))
                ->orderBy('initiatives.status', 'ASC')
                ->orderBy('initiatives.created_at', 'DESC');
    } elseif ($status) {
        // Regular status filter
        $builder->where('initiatives.status', $status)
                ->orderBy('initiatives.status', 'ASC')
                ->orderBy('initiatives.created_at', 'DESC');
    } else {
        // Default fetch
        $builder->orderBy('initiatives.status', 'ASC')
                ->orderBy('initiatives.created_at', 'DESC');
    }

    $initiatives = $builder->findAll();

    if ($action === 'download') {
        return view('downloads/all_initiatives', [
            'initiatives' => $initiatives,
            'organization' => $this->organization
        ]);
    }

    return view('initiatives/index', [
        'initiatives' => $initiatives,
        'organization' => $this->organization
    ]);
}


    public function goalInitiatives($slug)
    {
        $goal = $this->goalsModel->where('slug', $slug)->first();

        $rawInitiatives = $this->initiativesModel->select('
        initiatives.*,
        initiativeowners.id AS owner_link_id,
        initiativeowners.status AS owner_status,
        users.name AS user_owner_name,
        organizationstructures.name AS org_owner_name,
        kpis.name AS kpi_name,
        kpis.status AS kpi_status')
            ->join('initiativeowners', 'initiativeowners.initiative_id = initiatives.id', 'left')
            ->join('users', 'users.id = initiativeowners.individual_owner_id', 'left')
            ->join('organizationstructures', 'organizationstructures.id = initiativeowners.organizationstructure_owner_id', 'left')
            ->join('kpis', 'kpis.initiative_id = initiatives.id', 'left')
            ->where('initiatives.goal_id', $goal['id'])
            ->orderBy('initiatives.status')
            ->orderBy('initiatives.created_at', 'DESC')
            ->get()->getResultArray();

        $groupedInitiatives = [];

        foreach ($rawInitiatives as $row) {
            $id = $row['id'];

            if (!isset($groupedInitiatives[$id])) {
                $groupedInitiatives[$id] = $row;
                $groupedInitiatives[$id]['owners'] = [];
                $groupedInitiatives[$id]['kpis'] = [];
            }

            // Only include owner if status is active
            if ($row['owner_status'] === 'active') {
                $ownerName = $row['org_owner_name'] ?: $row['user_owner_name'];
                $ownerId = $row['owner_link_id'];

                if ($ownerName && $ownerId) {
                    $alreadyAdded = false;
                    foreach ($groupedInitiatives[$id]['owners'] as $existingOwner) {
                        if ($existingOwner['id'] == $ownerId) {
                            $alreadyAdded = true;
                            break;
                        }
                    }

                    if (!$alreadyAdded) {
                        $groupedInitiatives[$id]['owners'][] = [
                            'id' => $ownerId,
                            'name' => $ownerName
                        ];
                    }
                }
            }

            // Only add KPI if its status is active
            if (
                !empty($row['kpi_name']) &&
                $row['kpi_status'] === 'active' &&
                !in_array($row['kpi_name'], $groupedInitiatives[$id]['kpis'])
            ) {
                $groupedInitiatives[$id]['kpis'][] = $row['kpi_name'];
            }
        }

        $initiatives = array_values($groupedInitiatives);
        // dd($initiatives);

        //return file to download initiatives as pdf or excel
        $action = $this->request->getGet('action');
        if ($action === 'download') {
            return view('downloads/initiatives', [
                'organization' => $this->organization,
                'goal' => $goal,
                'initiatives' => $initiatives
            ]);
        }

        return view('initiatives/goal_initiatives', [
            'organization' => $this->organization,
            'goal' => $goal,
            'initiatives' => $initiatives
        ]);
    }




    // diaplay form to create initiatives
    public function create($slug)
    {
        $goal = $this->goalsModel->where('slug', $slug)->first();

        return view('initiatives/create', [
            'organization' => $this->organization,
            'goal' => $goal,
        ]);
    }



    // create new initiative
    public function store()
    {
        $rules = [
            'name' => 'required',
            'target' => 'required|max_length[255]',
            'measure_of_success' => 'required',
            'start_date' => 'required|valid_date',
            'end_date'   => 'required|valid_date|valid_date_range',
        ];

        $messages = [
            'name' => [
                'required' => 'Provide initiative name',
            ],
            'measure_of_success' => [
                'required' => 'Choose a measure of success',
            ],
            'start_date' => [
                'required' => 'Choose the start date',
            ],
            'end_date' => [
                'required' => 'Choose the expected date of completion',
                'valid_date_range' => 'End date must be after the start date',
            ],
        ];

        if (!$this->validate($rules, $messages)) {
            return view('initiatives/create', ['validation' => $this->validator, 'goal' => $this->goalsModel->where('id', $this->request->getPost('id'))->first()]);
        }

        $data = [
            'organization_id' => $this->organization['id'],
            'goal_id' => $this->request->getPost('id'),
            'slug' => substr(md5(uniqid(time(), true)), 0, 32),
            'name' => $this->request->getPost('name'),
            'target' => $this->request->getPost('target'),
            'measure_of_success' => $this->request->getPost('measure_of_success'),
            'start_date' => $this->request->getPost('start_date'),
            'end_date' => $this->request->getPost('end_date'),
        ];

        if ($this->initiativesModel->save($data)) {
            $goal = $this->goalsModel->where('id', $data['goal_id'])->first();

            return redirect()->to("goal/" . $goal['slug'] . "/initiatives")->with('success', 'Initiative created successfully');
        }
    }

    // display form to edit initiatvies
    public function edit($slug)
    {
        $initiative = $this->initiativesModel->where('slug', $slug)->first();
        // dd($initiative);
        return view('initiatives/edit', ['organization' => $this->organization, 'initiative' => $initiative]);
    }

    // collect data and update initiative
    public function update()
    {
        $rules = [
            'name' => 'required',
            'target' => 'required|max_length[255]',
            'measure_of_success' => 'required',
            'start_date' => 'required|valid_date',
            'end_date'   => 'required|valid_date|valid_date_range',
        ];

        $messages = [
            'name' => [
                'required' => 'Provide initiative name',
            ],
            'measure_of_success' => [
                'required' => 'Choose a measure of success',
            ],
            'start_date' => [
                'required' => 'Choose the start date',
            ],
            'end_date' => [
                'required' => 'Choose the expected date of completion',
                'valid_date_range' => 'End date must be after the start date',
            ],
        ];

        $id = $this->request->getPost('id');
        if (!$this->validate($rules, $messages)) {
            $initiative = $this->initiativesModel->where('id', $id)->first();
            return view('initiatives/edit', ['organization' => $this->organization, 'initiative' => $initiative, 'validation' => $this->validator, 'goal' => $this->goalsModel->where('id', $id)->first()]);
        }

        $data = [
            'name' => $this->request->getPost('name'),
            'target' => $this->request->getPost('target'),
            'measure_of_success' => $this->request->getPost('measure_of_success'),
            'start_date' => $this->request->getPost('start_date'),
            'end_date' => $this->request->getPost('end_date'),
        ];

        if ($this->initiativesModel->update($id, $data)) {
            $initiative = $this->initiativesModel->where('id', $id)->first();
            $goal = $this->goalsModel->where('id', $initiative['goal_id'])->first();

            return redirect()->to("goal/" . $goal['slug'] . "/initiatives")->with('success', 'Initiative updated successfully');
        }
    }

    // disable initiative
    public function disable($id)
    {
        // dd('ok');
        $data = [
            'status' => 'inactive',
        ];
        $previousUrl = previous_url(true);
        // dd($goal);
        if ($this->initiativesModel->update($id, $data)) {

            return redirect()->to($previousUrl)->with('success', 'Initiative disabled successfully');
        }
        return redirect()->to($previousUrl)->with('error', 'Unable to disable initiative');
    }

    // enable initiative
    public function enable($id)
    {
        // dd('enable');
        $data = [
            'status' => 'active',
        ];
        $previousUrl = previous_url(true);
        // dd($goal);
        if ($this->initiativesModel->update($id, $data)) {

            return redirect()->to($previousUrl)->with('success', 'Initiative enabled successfully');
        }
        return redirect()->to($previousUrl)->with('error', 'Unable to enabled initiative');
    }

    // mark initiative as completed
    public function markComplete($id)
    {
        // dd('enable');
        $data = [
            'current_status' => 'completed',
            'rag_status' => 'green',
            'status' => 'completed',
        ];
        $previousUrl = previous_url(true);
        // dd($goal);
        if ($this->initiativesModel->update($id, $data)) {

            return redirect()->to($previousUrl)->with('success', 'Initiative marked complete');
        }
        return redirect()->to($previousUrl)->with('error', 'Unable to mark initiative complete');
    }

    // fetch initiative owners
    public function getInitiativeOwners($initiativeid)
    {
        $owners = $this->initiativesOwnerModel
            ->select('
            initiativeowners.id,
            initiativeowners.status,
            users.name AS user_owner_name,
            organizationstructures.name AS org_owner_name
        ')
            ->join('users', 'users.id = initiativeowners.individual_owner_id', 'left')
            ->join('organizationstructures', 'organizationstructures.id = initiativeowners.organizationstructure_owner_id', 'left')
            ->where('initiativeowners.initiative_id', $initiativeid)
            ->orderBy('initiativeowners.status')
            ->findAll();

        // Optionally merge the name into one field
        foreach ($owners as &$owner) {
            $owner['owner_name'] = $owner['user_owner_name'] ?? $owner['org_owner_name'];
        }

        return $owners;
    }

    // fetch initiative owners and send to owners view
    public function initiative_owners($slug)
    {
        $initiative = $this->initiativesModel->where('slug', $slug)->first();

        return view('initiatives/owners', [
            'organization' => $this->organization,
            'initiative' => $initiative,
            'owners' => $this->getInitiativeOwners($initiative['id']),
        ]);
    }


    // display form to assign initiative owner
    public function initiative_owner_form($slug)
    {
        $initiative = $this->initiativesModel->where('slug', $slug)->first();
        $structures = $this->organizationStructureModel->where('organization_id', $this->organization['id'])->where('status', 'active')->findAll();
        $users = $this->userModel->where('organizations_id', $this->organization['id'])->where('status', 'active')->findAll();

        return view('initiatives/assign_owner', ['organization' => $this->organization, 'initiative' => $initiative, 'structures' => $structures, 'users' => $users]);
    }

    public function assign_initiative_owner()
    {
        $rules = [
            'initiative_id' => 'atLeastOneOwner' //custom validation rule
        ];

        $messages = [
            'initiative_id' => [
                'atLeastOneOwner' => 'Choose a unit or an individual as the owner',
            ],
        ];

        $data = [
            'organization_id' => $this->organization['id'],
            'initiative_id' => $this->request->getPost('id'),
            'organizationstructure_owner_id' => $this->request->getPost('organizationstructure_owner_id') ?: null,
            'individual_owner_id' => $this->request->getPost('individual_owner_id') ?: null,
        ];

        $initiative = $this->initiativesModel->where('id', $data['initiative_id'])->first();
        $structures = $this->organizationStructureModel->where('organization_id', $this->organization['id'])->where('status', 'active')->findAll();
        $users = $this->userModel->where('organizations_id', $this->organization['id'])->where('status', 'active')->findAll();

        if (!$this->validate($rules, $messages)) {
            return view('initiatives/assign_owner', ['organization' => $this->organization, 'validation' => $this->validator, 'initiative' => $initiative, 'structures' => $structures, 'users' => $users]);
        }

        $org_Owner = $this->initiativesOwnerModel->where('initiative_id', $data['initiative_id'])->where('organizationstructure_owner_id', $data['organizationstructure_owner_id'])->first();
        $ind_Owner = $this->initiativesOwnerModel->where('initiative_id', $data['initiative_id'])->where('individual_owner_id', $data['individual_owner_id'])->first();

        if ($org_Owner && $ind_Owner) {
            return redirect()->to('initiatives/' . $initiative['slug'] . '/owners')->with('error', 'Owner already assigned to initiative');
        }

        if ($this->initiativesOwnerModel->save($data)) {
            return redirect()->to('initiatives/' . $initiative['slug'] . '/owners')->with('success', 'Owner assigned successfully');
        }
    }

    // disable initiative owner
    public function disable_initiative_owner($id)
    {
        $data = [
            'status' => 'inactive',
        ];

        if ($this->initiativesOwnerModel->update($id, $data)) {
            $owner = $this->initiativesOwnerModel->where('id', $id)->first();
            $initiative = $this->initiativesModel->where('id', $owner['initiative_id'])->first();

            return redirect()->to('initiatives/' . $initiative['slug'] . '/owners')->with('success', 'Owner disabled successfully');
        }
    }

    // disable initiative owner
    public function enable_initiative_owner($id)
    {
        $data = [
            'status' => 'active',
        ];

        if ($this->initiativesOwnerModel->update($id, $data)) {
            $owner = $this->initiativesOwnerModel->where('id', $id)->first();
            $initiative = $this->initiativesModel->where('id', $owner['initiative_id'])->first();

            return redirect()->to('initiatives/' . $initiative['slug'] . '/owners')->with('success', 'Owner disabled successfully');
        }
    }

    // public function remove_initiative_owner($id)
    // {
    //     $owner = $this->initiativesOwnerModel->where('id', $id)->first();
    //     $initiative = $this->initiativesModel->where('id', $owner['initiative_id'])->first();
    //     $goal = $this->goalsModel->where('id', $initiative['goal_id'])->first();
    //     // dd($initiative);
    //     if ($this->initiativesOwnerModel->delete($id)) {
    //         return redirect()->to("goals/" . $goal['slug'] . "/initiatives")->with('success', 'Owner removed successfully');
    //     }
    // }

    // fetch intiatives based on owner
    public function initiativesOwner()
    {
        $id = session()->get('user_id');
        $user = $this->userModel->where('id', $id)->first();

        $builder = $this->initiativesModel->select('
        initiatives.*,
        initiativeowners.id AS owner_link_id,
        users.name AS user_owner_name,
        organizationstructures.name AS org_owner_name
    ')
            ->join('initiativeowners', 'initiativeowners.initiative_id = initiatives.id', 'left')
            ->join('users', 'users.id = initiativeowners.individual_owner_id', 'left')
            ->join('organizationstructures', 'organizationstructures.id = initiativeowners.organizationstructure_owner_id', 'left')
            ->groupStart()
            ->where('initiativeowners.individual_owner_id', $id);

        // Only add organization structure check if the user has one
        if (!empty($user['organizationstructures_id'])) {
            $builder->orWhere('initiativeowners.organizationstructure_owner_id', $user['organizationstructures_id']);
        }

        $builder->groupEnd();

        $initiatives = $builder->get()->getResultArray();

        return view('users/initiatives', ['initiatives' => $initiatives]);
    }

    // get a user's initiatives
    public function getUserInitiatives($slug, $status = null)
    {
        $userslug = $slug ?? session()->get('user_slug');
        $user = $this->userModel->where('slug', $userslug)->first();
        $userOrgStructureId = $user['organizationstructures_id'] ?? null;

        $builder = $this->initiativesModel->builder();
        $builder->select('initiatives.*');
        $builder->join('initiativeowners', 'initiativeowners.initiative_id = initiatives.id');
        $builder->where('initiativeowners.status', 'active');

        // Include only initiatives owned by this user or their department
        $builder->groupStart()
            ->where('initiativeowners.individual_owner_id', $user['id']);

        if ($userOrgStructureId !== null) {
            $builder->orWhere('initiativeowners.organizationstructure_owner_id', $userOrgStructureId);
        }

        $builder->groupEnd();

        // Add missing organization filter
        $builder->where('initiatives.organization_id', $this->organization['id']);

        // status filtering
        if (!is_null($status)) {
            if ($status === 'due') {
                // Match logic from index(): due = end_date < now and not completed/inactive
                $builder->where('initiatives.end_date <', date('Y-m-d H:i:s'))
                    ->whereNotIn('initiatives.status', ['completed', 'inactive']);
            } else {
                $builder->where('initiatives.status', $status);
            }
        }

        $builder->distinct();
        $builder->orderBy('initiatives.status');

        return $builder->get()->getResultArray();
    }



    public function user_initiatives($slug)
    {
        $initiatives = $this->getUserInitiatives($slug, $this->request->getGet('status'));

        return view('users/owned/initiatives', [
            'organization' => $this->organization,
            'initiatives' => $initiatives,
        ]);
    }
}
