<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\GoalsModel;
use App\Models\InitiativesModel;
use App\Models\KPIModel;
use App\Models\KPIOwnersModel;
use App\Models\OrganizationsModel;
use App\Models\OrganizationStructureModel;
use App\Models\UserModel;
use CodeIgniter\HTTP\ResponseInterface;

class KPIsController extends BaseController
{
    protected $initiativesModel;
    protected $kpisModel;
    protected $goalsModel;
    protected $organizationsModel;
    protected $organization;
    protected $organizationStructureModel;
    protected $userModel;
    protected $kpiOwnersModel;

    public function __construct()
    {
        $this->initiativesModel = new InitiativesModel();
        $this->kpisModel = new KPIModel();
        $this->goalsModel = new GoalsModel();
        $this->organizationsModel = new OrganizationsModel();
        $this->organization = $this->organizationsModel->where('slug', session()->get('slug'))->first();
        $this->organizationStructureModel = new OrganizationStructureModel();
        $this->userModel = new UserModel();
        $this->kpiOwnersModel = new KPIOwnersModel();
    }
    
    public function all_kpis()
    {
        $kpis = $this->kpisModel->where('organization_id', $this->organization['id'])->orderBy('status')->orderBy('created_at', 'DESC')->findAll();

        // Fetch and attach owners to each KPI
        foreach ($kpis as &$kpi) {
            $kpi['owners'] = $this->getKPIOwners($kpi['id']);
        }

        // dd($kpis);
        return view('kpis/all_kpis', ['organization' => $this->organization, 'kpis' => $kpis]);
    }
    
    // upload bulk kpis
    public function storeBulk()
    {

        helper(['form', 'filesystem']);
    
        // Ensure a file was uploaded
        $file = $this->request->getFile('kpis');
        
        if (!$file || !$file->isValid() || $file->hasMoved()) {
            return redirect()->back()->with('error', 'Please upload a valid file.');
        }
    
        // Validate file extension
        $extension = $file->getExtension();
        $allowedExtensions = ['csv', 'xls', 'xlsx'];
        
        if (!in_array($extension, $allowedExtensions)) {
            return redirect()->back()->with('error', 'Only CSV and Excel files (.csv, .xls, .xlsx) are allowed.');
        }
    
        try {
            // Get file path
            $filePath = $file->getTempName();
    
            // Read file based on extension using PhpSpreadsheet
            if ($extension === 'csv') {
                $reader = new \PhpOffice\PhpSpreadsheet\Reader\Csv();
                $reader->setDelimiter(','); // or ';' if your file uses semicolons
                $reader->setEnclosure('"');
                $reader->setSheetIndex(0);
            } elseif ($extension === 'xlsx') {
                $reader = new \PhpOffice\PhpSpreadsheet\Reader\Xlsx();
            } else { // xls
                $reader = new \PhpOffice\PhpSpreadsheet\Reader\Xls();
            }
    
            // Load spreadsheet
            $spreadsheet = $reader->load($filePath);
            $sheetData = $spreadsheet->getActiveSheet()->toArray();
                
            var_dump($sheetData);
            exit;
            // Remove header row (first row)
            array_shift($sheetData);
    
            $inserted = 0;
            $skipped = 0;
            $errors = [];
    
            foreach ($sheetData as $index => $row) {
                $rowNumber = $index + 2; // +2 because array index starts at 0 and we removed header
    
                // Skip empty rows
                if (empty(array_filter($row))) {
                    continue;
                }
    
                // Validate row has minimum required columns
                if (count($row) < 2 || empty($row[0]) || empty($row[1])) {
                    $skipped++;
                    $errors[] = "Row {$rowNumber}: Missing required data (initiative or kpi)";
                    continue;
                }
    
                $initiativeId = trim($row[0]);
                $initiativeId = (int)$initiativeId; // normalize numeric ID
                $kpiName = trim($row[1]);
                $timeline = isset($row[2]) && !empty($row[2]) ? trim($row[2]) : null;

    
                // Verify initiative belongs to the same organization
                $initiative = $this->initiativesModel
                    ->where('id', $initiativeId)
                    ->where('organization_id', $this->organization['id'])
                    ->first();
    
                if (!$initiative) {
                    $skipped++;
                    $errors[] = "Row {$rowNumber}: Initiative {$initiativeId} not found or doesn't belong to your organization";
                    continue;
                }
    
                // Validate and format timeline if provided
                $formattedTimeline = null;
                if ($timeline) {
                    // Handle Excel date serial numbers
                    if (is_numeric($timeline)) {
                        try {
                            $formattedTimeline = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($timeline)->format('Y-m-d');
                        } catch (\Exception $e) {
                            $skipped++;
                            $errors[] = "Row {$rowNumber}: Invalid date format for timeline";
                            continue;
                        }
                    } else {
                        // Handle string dates
                        $timestamp = strtotime($timeline);
                        if ($timestamp === false) {
                            $skipped++;
                            $errors[] = "Row {$rowNumber}: Invalid date format for timeline";
                            continue;
                        }
                        $formattedTimeline = date('Y-m-d', $timestamp);
                    }
                }
    
                // Prepare KPI data
                $data = [
                    'organization_id' => $this->organization['id'],
                    'initiative_id' => $initiative['id'],
                    'slug' => substr(md5(uniqid(time(), true)), 0, 32),
                    'name' => $kpiName,
                    'timeline' => $formattedTimeline ?: $initiative['end_date'],
                    'status' => 'active'
                ];
    
                // Insert KPI
                if ($this->kpisModel->save($data)) {
                    $inserted++;
                } else {
                    $skipped++;
                    $errors[] = "Row {$rowNumber}: Failed to save KPI - " . implode(', ', $this->kpisModel->errors());
                }
            }
    
            // Prepare feedback message
            $msg = "Bulk upload completed. Successfully added {$inserted} KPI(s)";
            
            if ($skipped > 0) {
                $msg .= ", skipped {$skipped}";
            }
            
            $msgType = $inserted > 0 ? 'success' : 'warning';
    
            // Store errors in session if any (limit to first 10 to avoid session bloat)
            if (!empty($errors)) {
                session()->setFlashdata('upload_errors', array_slice($errors, 0, 10));
                if (count($errors) > 10) {
                    session()->setFlashdata('more_errors', count($errors) - 10);
                }
            }
    
            return redirect()->to('kpis')->with($msgType, $msg);
    
        } catch (\PhpOffice\PhpSpreadsheet\Reader\Exception $e) {
            log_message('error', 'PhpSpreadsheet error during bulk KPI upload: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error reading file: ' . $e->getMessage());
        } catch (\Exception $e) {
            log_message('error', 'Bulk KPI upload error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'An error occurred while processing the file. Please check the file format and try again.');
        }
    }
    
    public function index($slug)
    {
        $initiative = $this->initiativesModel->where('slug', $slug)->first();
        $kpis = $this->kpisModel->where('initiative_id', $initiative['id'])->orderBy('status')->orderBy('created_at', 'DESC')->findAll();

        // Fetch and attach owners to each KPI
        foreach ($kpis as &$kpi) {
            $kpi['owners'] = $this->getKPIOwners($kpi['id']);
        }

        // dd($kpis);
        return view('kpis/index', ['initiative' => $initiative, 'organization' => $this->organization, 'kpis' => $kpis]);
    }

    // display form to create KPIs
    public function create($slug)
    {
        $initiative = $this->initiativesModel->where('slug', $slug)->first();
        return view('kpis/create', ['initiative' => $initiative, 'organization' => $this->organization]);
    }

    // collect form data and create kpi
    public function store()
    {
        $rules = [
            'name' => 'required',
            // 'timeline' => 'required',
        ];

        $messages = [
            'name' => [
                'required' => 'Provide KPI name',
            ],
            'timeline' => [
                'required' => 'Specify expected date of completion',
            ],
        ];
        
        $initiative = $this->initiativesModel->where('id', $this->request->getPOst('id'))->first();

        $timeline = $this->request->getPost('timeline');
        $data = [
            'organization_id' => $this->organization['id'],
            'initiative_id' => $initiative['id'],
            'slug' => substr(md5(uniqid(time(), true)), 0, 32),
            'name' => $this->request->getPost('name'),
            'timeline' => !empty($timeline) ? $timeline : $initiative['end_date'],
        ];


        if (!$this->validate($rules, $messages)) {
            return view('kpis/create', ['organization' => $this->organization, 'initiative' => $initiative, 'validation' => $this->validator]);
        }

        if ($this->kpisModel->save($data)) {
            return redirect()->to("initiatives/" . $initiative['slug'] . "/kpis")->with('success', 'KPI added successfully');
        }
    }

    // diaplay form to update KPI
    public function edit($slug)
    {
        $kpi = $this->kpisModel->where('slug', $slug)->first();

        return view('kpis/edit', ['kpi' => $kpi, 'organization' => $this->organization]);
    }

    // collect form data and update kpi
    public function update()
    {
        $id = $this->request->getPost('id');
        $kpi = $this->kpisModel->where('id', $id)->first();

        $rules = [
            'name' => 'required',
            'timeline' => 'required',
        ];

        $messages = [
            'name' => [
                'required' => 'Provide KPI name',
            ],
            'timeline' => [
                'required' => 'Specify expected date of completion',
            ],
        ];

        $data = [
            'name' => $this->request->getPOst('name'),
            'timeline' => $this->request->getPOst('timeline'),
        ];
        $initiative = $this->initiativesModel->where('id', $kpi['initiative_id'])->first();

        if (!$this->validate($rules, $messages)) {
            return view('kpis/edit', ['organization' => $this->organization, 'initiative' => $initiative, 'validation' => $this->validator]);
        }

        if ($this->kpisModel->update($id, $data)) {
            return redirect()->to("initiatives/" . $initiative['slug'] . "/kpis")->with('success', 'KPI updated successfully');
        }
    }

    // disable kpi
    public function disable($id)
    {
        // dd('ok');
        $kpi = $this->kpisModel->where('id', $id)->first();
        $initiative = $this->initiativesModel->where('id', $kpi['initiative_id'])->first();

        $data = [
            'status' => 'inactive',
        ];

        if ($this->kpisModel->update($id, $data)) {

            return redirect()->to("initiatives/" . $initiative['slug'] . "/kpis")->with('success', 'KPI disabled successfully');
        }
        return redirect()->to("initiatives/" . $initiative['slug'] . "/kpis")->with('success', 'Unable to disable KPI');
    }

    // enable KPI
    public function enable($id)
    {
        $kpi = $this->kpisModel->where('id', $id)->first();
        $initiative = $this->initiativesModel->where('id', $kpi['initiative_id'])->first();

        $data = [
            'status' => 'active',
        ];

        if ($this->kpisModel->update($id, $data)) {

            return redirect()->to("initiatives/" . $initiative['slug'] . "/kpis")->with('success', 'KPI enabled successfully');
        }
        return redirect()->to("initiatives/" . $initiative['slug'] . "/kpis")->with('success', 'Unable to enable KPI');
    }

    // mark a kpi complete
    public function markComplete($id)
    {
        $kpi = $this->kpisModel->where('id', $id)->first();
        $initiative = $this->initiativesModel->where('id', $kpi['initiative_id'])->first();

        $data = [
            'status' => 'completed',
        ];

        if ($this->kpisModel->update($id, $data)) {
            return redirect()->to("initiatives/" . $initiative['slug'] . "/kpis")->with('success', 'KPI marked completed');
        }
    }

    // fetch kpi owners
    public function getKPIOwners($kpiId)
    {
        $owners = $this->kpiOwnersModel
            ->select('
            kpiowners.id,
            kpiowners.status,
            users.name AS user_owner_name,
            organizationstructures.name AS org_owner_name
        ')
            ->join('users', 'users.id = kpiowners.individual_owner_id', 'left')
            ->join('organizationstructures', 'organizationstructures.id = kpiowners.organizationstructure_owner_id', 'left')
            ->where('kpiowners.kpi_id', $kpiId)
            ->orderBy('kpiowners.status')
            ->findAll();

        // Optionally merge the name into one field
        foreach ($owners as &$owner) {
            $owner['owner_name'] = $owner['user_owner_name'] ?? $owner['org_owner_name'];
        }

        return $owners;
    }

    // fetch kpi owners
    public function kpi_owners($slug)
    {
        $kpi = $this->kpisModel->where('slug', $slug)->first();
        // dd($this->getKPIOwners($kpi['id']));

        return view('kpis/owners', ['organization' => $this->organization, 'kpi' => $kpi, 'owners' => $this->getKPIOwners($kpi['id'])]);
    }

    // display form to add kpi owners
    public function kpi_owner_form($slug)
    {
        $kpi = $this->kpisModel->where('slug', $slug)->first();
        $structures = $this->organizationStructureModel->where('organization_id', $this->organization['id'])->where('status', 'active')->findAll();
        $users = $this->userModel->where('organizations_id', $this->organization['id'])->where('status', 'active')->findAll();

        return view('kpis/assign_owner', ['kpi' => $kpi, 'structures' => $structures, 'users' => $users, 'organization' => $this->organization]);
    }

    // assign kpi owner
    public function assign_kpi_owner()
    {
        $rules = [
            'kpi_id' => 'atLeastOneOwner' //custom validation rule
        ];

        $messages = [
            'kpi_id' => [
                'atLeastOneOwner' => 'Choose a unit or an individual as the owner',
            ],
        ];

        $data = [
            'organization_id' => $this->organization['id'],
            'kpi_id' => $this->request->getPost('id'),
            'organizationstructure_owner_id' => $this->request->getPost('organizationstructure_owner_id') ?: null,
            'individual_owner_id' => $this->request->getPost('individual_owner_id') ?: null,
        ];

        $kpi = $this->kpisModel->where('id', $data['kpi_id'])->first();
        $structures = $this->organizationStructureModel->where('organization_id', $this->organization['id'])->where('status', 'active')->findAll();
        $users = $this->userModel->where('organizations_id', $this->organization['id'])->where('status', 'active')->findAll();

        if (!$this->validate($rules, $messages)) {
            return view('kpis/assign_owner', ['organization' => $this->organization, 'validation' => $this->validator, 'kpi' => $kpi, 'structures' => $structures, 'users' => $users]);
        }

        $org_Owner = $this->kpiOwnersModel->where('kpi_id', $data['kpi_id'])->where('organizationstructure_owner_id', $data['organizationstructure_owner_id'])->first();
        $ind_Owner = $this->kpiOwnersModel->where('kpi_id', $data['kpi_id'])->where('individual_owner_id', $data['individual_owner_id'])->first();

        if ($org_Owner && $ind_Owner) {
            return redirect()->to('initiatives/kpi/' . $kpi['slug'] . '/owners')->with('error', 'Owner already assigned to KPI');
        }

        if ($this->kpiOwnersModel->save($data)) {
            return redirect()->to('initiatives/kpi/' . $kpi['slug'] . '/owners')->with('success', 'Owner assigned successfully');
        }
    }

    // disable kpi owner
    public function disable_kpi_owner($id)
    {
        $data = [
            'status' => 'inactive',
        ];

        if ($this->kpiOwnersModel->update($id, $data)) {
            $owner = $this->kpiOwnersModel->where('id', $id)->first();
            // dd($owner);
            $kpi = $this->kpisModel->where('id', $owner['kpi_id'])->first();

            return redirect()->to('initiatives/kpi/' . $kpi['slug'] . '/owners')->with('success', 'Owner disabled successfully');
        }
    }

    // disable kpi owner
    public function enable_kpi_owner($id)
    {
        $data = [
            'status' => 'active',
        ];

        if ($this->kpiOwnersModel->update($id, $data)) {
            $owner = $this->kpiOwnersModel->where('id', $id)->first();
            $kpi = $this->kpisModel->where('id', $owner['kpi_id'])->first();

            return redirect()->to('initiatives/kpi/' . $kpi['slug'] . '/owners')->with('success', 'Owner disabled successfully');
        }
    }

    public function getUserKpis($slug, $status = null)
    {
        $userslug = $slug ?? session()->get('user_slug');
        $user = $this->userModel->where('slug', $userslug)->first();
        $userOrgStructureId = $user['organizationstructures_id'] ?? null;

        $builder = $this->kpisModel->builder();
        $builder->select('kpis.*');
        $builder->join('kpiowners', 'kpiowners.kpi_id = kpis.id');
        $builder->where('kpiowners.status', 'active');
        $builder->orderBy('kpis.status');

        // Apply status filter
        if (!is_null($status)) {
            if ($status === 'due') {
                $builder->where('kpis.timeline <', date('Y-m-d'))
                    ->whereNotIn('kpis.status', ['completed', 'inactive']);
            } else {
                $builder->where('kpis.status', $status);
            }
        }

        // Ownership logic
        $builder->groupStart()
            ->where('kpiowners.individual_owner_id', $user['id']);

        if ($userOrgStructureId !== null) {
            $builder->orWhere('kpiowners.organizationstructure_owner_id', $userOrgStructureId);
        }

        $builder->groupEnd();

        $builder->distinct();

        $kpis = $builder->get()->getResultArray();
        return $kpis;
    }

    public function user_kpis($slug)
    {
        $kpis = $this->getUserKpis($slug, $this->request->getGet('status'));
        // dd($kpis);

        return view('users/owned/kpis', [
            'organization' => $this->organization,
            'kpis' => $kpis,
        ]);
    }
}
