<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\OrganizationsModel;
use App\Models\UserModel;
use App\Models\SubscriptionModel;
use CodeIgniter\HTTP\ResponseInterface;

class OrganizationsController extends BaseController
{
    protected $organizationsModel;
    protected $email;
    protected $userModel;
    protected $subscriptionModel;

    public function __construct()
    {
        $this->organizationsModel = new OrganizationsModel();
        $this->email = new \App\Controllers\EmailController;
        $this->userModel = new UserModel();
        $this->subscriptionModel = new SubscriptionModel();
    }

     public function index()
    {
        $status = $this->request->getGet('status');
    
        $builder = $this->organizationsModel
            ->orderBy('status', 'asc')
            ->orderBy('created_at', 'desc');
    
        // Apply status filter if provided
        if (!is_null($status)) {
            $builder->where('status', $status);
        }
    
        $organizations = $builder->findAll();
    
        foreach ($organizations as &$organization) {
            // Count total users under organization
            $users = $this->userModel->where('organizations_id', $organization['id'])->findAll();
            $organization['total_users'] = count($users);
    
            // Fetch the current active subscription for this organization
            $activeSub = $this->subscriptionModel
                ->select('plan, start_date, end_date')
                ->where('organization_id', $organization['id'])
                ->where('end_date >=', date('Y-m-d'))
                ->orderBy('end_date', 'desc')
                ->first();
    
            // Attach active subscription data if available
            if ($activeSub) {
                $organization['active_subscription'] = [
                    'plan'       => $activeSub['plan'],
                    'start_date' => date('Y-m-d', strtotime($activeSub['start_date'])),
                    'end_date'   => date('Y-m-d', strtotime($activeSub['end_date'])),
                ];
            } else {
                $organization['active_subscription'] = null;
            }
        }
    
        return view('organizations/index', ['organizations' => $organizations]);
    }

    // display form to create new oranizations
    public function create()
    {
        return view('organizations/create');
    }

    // collect form data and create new organizations
    public function new()
    {
        $validationRules = [
            'name' => 'required|max_length[255]|string',
            'email' => 'required|valid_email|is_unique[organizations.email]',
            'max_users' => 'required',
        ];

        $validationMessages = [
            'name' => [
                'required' => 'The name field is required.',
                'string'   => 'The name must be a valid string.',
            ],
            'email' => [
                'required'    => 'The email field is required.',
                'valid_email' => 'Please provide a valid email address.',
                'is_unique'   => 'This email has already been taken.'
            ],
            'max_users' => [
                'required'    => 'Specify maximum number of users',
            ],
        ];

        if (!$this->validate($validationRules, $validationMessages)) {
            return view('organizations/create', ['validation' => $this->validator]);
        }

        $passcode = substr(str_shuffle(bin2hex(random_bytes(4)) . 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz'), 0, 8);
        $hashedPassword = password_hash($passcode, PASSWORD_DEFAULT);

        $data = [
            'slug' => substr(md5(uniqid(time(), true)), 0, 40),
            'name' => $this->request->getPost('name'),
            'email' => $this->request->getPost('email'),
            'max_users' => $this->request->getPost('max_users'),
            'password' => $hashedPassword,
            'passcode' => $passcode,
        ];

        // email details
        // Create link
        $link = base_url("login");

        $subject = 'New STracker Account';
        $message = "<p style='margin-top:8px;'>Dear {$data['name']},</p>
            <p style='margin-top:8px;'>An account has been created for your organization on STracker with the following details:</p>
            <p style='margin-top:8px;'><strong>Email:</strong> {$data['email']}</p>
            <p style='margin-top:8px;'><strong>Password:</strong> {$data['passcode']}</p>
            <p style='margin-top:8px;'>Click on the link below to login.</p>
            <p style='margin: 18px 0; text-align: center;'>
                <a href=\"$link\" style='background-color:#51585e; text-decoration:none; padding:12px 24px; border-radius:6px; color:white; display:inline-block;'>
                    Login
                </a>
            </p>
            <p style='margin-top:8px;'>If the above button does not work, copy this link and paste in your url address bar $link</p>
            <strong style='margin-top:12px;'>Best Regards.</strong>
            <div style='margin-top:12px; text-align: center; background-color:rgb(233, 233, 233); font-size:12px; padding:12px'>STracker | Suite 3F4, Katu Koma Plaza, 52 Off Oladipo Diya Way, Sector Centre C, Abuja, FCT, Nigeria.</div>
            <h3 style='margin-top:8px; text-align: center;'>&copy; " . date('Y') . " Learning Impact</h3>";


        if ($this->organizationsModel->save($data)) {
            $organization = $this->organizationsModel->where('email', $data['email'])->first();

            $userData = [
                'organizations_id' => $organization['id'],
                'slug' => substr(md5(uniqid(time(), true)), 0, 40),
                'role' => 'admin',
                'name' => $data['name'],
                'email' => $data['email'],
                'password' => $hashedPassword,
                'passcode' => $passcode,
            ];

            $this->userModel->save($userData);
            $this->email->sendMail($data['email'], $subject, $message);

            return redirect()->to('organizations')->with('success', 'Account created successfully');
        }
    }

    // disable organization
    public function disable($id)
    {
        // dd('ok');
        $data = [
            'status' => 'inactive',
        ];

        if ($this->organizationsModel->update($id, $data)) {
            return redirect()->to('organizations')->with('success', 'Organization disabled successfully');
        }
        return redirect()->to('organizations')->with('error', 'Unable to disable organization');
    }

    // enable organization
    public function enable($id)
    {
        // dd('enable');
        $data = [
            'status' => 'active',
        ];

        if ($this->organizationsModel->update($id, $data)) {
            return redirect()->to('organizations')->with('success', 'Organization enabled successfully');
        }
        return redirect()->to('organizations')->with('error', 'Unable to disable organization');
    }

    public function edit($slug)
    {
        $organization = $this->organizationsModel->where('slug', $slug)->first();

        return view('organizations/edit', ['organization' => $organization]);
    }

    public function update()
    {
        $id = $this->request->getPost('id');
        $organization = $this->organizationsModel->find($id);

        $validationRules = [
            'name' => 'required|max_length[255]|string',
            'email' => "required|valid_email|is_unique[organizations.email,id,{$id}]",
            'max_users' => 'required',
        ];

        $validationMessages = [
            'name' => [
                'required' => 'The name field is required.',
                'string'   => 'The name must be a valid string.',
            ],
            'email' => [
                'required'    => 'The email field is required.',
                'valid_email' => 'Please provide a valid email address.',
                'is_unique'   => 'This email has already been taken.'
            ],
            'max_users' => [
                'required'    => 'Specify maximum number of users',
            ],
        ];

        if (!$this->validate($validationRules, $validationMessages)) {
            return view('organizations/edit', [
                'validation' => $this->validator,
                'organization' => $organization
            ]);
        }

        $data = [
            'name' => $this->request->getPost('name'),
            'email' => $this->request->getPost('email'),
            'max_users' => $this->request->getPost('max_users'),
        ];

        if ($this->organizationsModel->update($id, $data)) {
            return redirect()->to('organizations')->with('success', 'Organization details updated successfully');
        }
        return redirect()->to('organizations')->with('error', 'Unable to update organization details');
    }
}
