<?php

namespace App\Controllers;

use App\Models\SubscriptionModel;
use App\Models\OrganizationsModel;
use App\Controllers\BaseController;
use CodeIgniter\HTTP\ResponseInterface;

class SubscriptionController extends BaseController
{
    protected $organizationsModel;
    protected $subscriptionModel;

    public function __construct()
    {
        $this->organizationsModel = new OrganizationsModel();
        $this->subscriptionModel = new SubscriptionModel();
    }

    public function index()
    {
        $status = $this->request->getGet('status');
    
        $builder = $this->subscriptionModel
            ->select('subscriptions.*, organizations.name as organization_name')
            ->join('organizations', 'organizations.id = subscriptions.organization_id', 'left')
            ->orderBy('subscriptions.status', 'asc')
            ->orderBy('subscriptions.created_at', 'desc');
    
        // Apply status filter if provided
        if (!is_null($status)) {
            if ($status === 'expired') {
                // Subscriptions whose end_date is before today
                $builder->where('subscriptions.end_date <', date('Y-m-d'));
            } elseif ($status === 'active') {
                // Active ones — end_date not passed yet
                $builder->where('subscriptions.end_date >=', date('Y-m-d'));
            } else {
                // If status is any other value, match directly
                $builder->where('subscriptions.status', $status);
            }
        }
    
        $subscriptions = $builder->findAll();
    
        return view('subscriptions/index', ['subscriptions' => $subscriptions]);
    }
    
    // display form to create sub for an organization
    public function create()
    {
        $organizations = $this->organizationsModel->where('status', 'active')->findAll();
        
        return view('subscriptions/create', ['organizations' => $organizations]);
    }
    
    // collect subscription data and store in db
    public function store()
    {
        $rules = [
            'organization_id' => [
                'rules' => 'required',
                'errors' => [
                    'required' => 'Please select an organization',
                ],
            ],
            'plan' => [
                'rules' => 'required',
                'errors' => [
                    'required' => 'Please choose a plan',
                ],
            ],
            'start_date' => [
                'rules' => 'required|valid_date',
                'errors' => [
                    'required' => 'Start date is required',
                    'valid_date' => 'Please enter a valid date',
                ],
            ],
            'end_date' => [
                'rules' => 'required|valid_date',
                'errors' => [
                    'required' => 'End date is required',
                    'valid_date' => 'Please enter a valid date',
                ],
            ],
        ];
        
        if(!$this->validate($rules))
        {
            $organizations = $this->organizationsModel->where('status', 'active')->findAll();
        
            return view('subscriptions/create', ['organizations' => $organizations, 'validation' => $this->validator]);
        }
        
        $data = [
            'slug'            => substr(md5(uniqid((string) time(), true)), 0, 32),
            'organization_id' => $this->request->getPost('organization_id'),
            'plan' => $this->request->getPost('plan'),
            'start_date' => $this->request->getPost('start_date'),
            'end_date' => $this->request->getPost('end_date'),
        ];
        
        if($this->subscriptionModel->save($data))
        {
            return redirect()->to('subscriptions')->with('success', 'Subscription created successfully');
        }
    }
    
    // display form to update sub
    public function edit($slug)
    {
        $subscription = $this->subscriptionModel->where('slug', $slug)->first();
        $organizations = $this->organizationsModel->where('status', 'active')->findAll();
        
        return view('subscriptions/edit', ['subscription' => $subscription, 'organizations' => $organizations]);
    }
    
    public function update($slug)
    {
        $rules = [
            'organization_id' => [
                'rules' => 'required',
                'errors' => [
                    'required' => 'Please select an organization',
                ],
            ],
            'plan' => [
                'rules' => 'required',
                'errors' => [
                    'required' => 'Please choose a plan',
                ],
            ],
            'start_date' => [
                'rules' => 'required|valid_date',
                'errors' => [
                    'required' => 'Start date is required',
                    'valid_date' => 'Please enter a valid date',
                ],
            ],
            'end_date' => [
                'rules' => 'required|valid_date',
                'errors' => [
                    'required' => 'End date is required',
                    'valid_date' => 'Please enter a valid date',
                ],
            ],
        ];
    
        // Find the subscription to update
        $subscription = $this->subscriptionModel->where('slug', $slug)->first();
    
        if (!$subscription) {
            return redirect()->to('subscriptions')->with('error', 'Subscription not found.');
        }
    
        // If validation fails, reload the edit page with errors
        if (!$this->validate($rules)) {
            $organizations = $this->organizationsModel->where('status', 'active')->findAll();
    
            return view('subscriptions/edit', [
                'subscription' => $subscription,
                'organizations' => $organizations,
                'validation' => $this->validator,
            ]);
        }
    
        // Prepare update data
        $data = [
            'organization_id' => $this->request->getPost('organization_id'),
            'plan' => $this->request->getPost('plan'),
            'start_date' => $this->request->getPost('start_date'),
            'end_date' => $this->request->getPost('end_date'),
        ];
    
        // Perform update
        if($this->subscriptionModel->update($subscription['id'], $data))
        {
            return redirect()->to('subscriptions')->with('success', 'Subscription updated successfully');
        }
    }
    
    // disable subscriptions
    public function markExpired($id)
    {
        // dd('ok');
        $data = [
            'status' => 'expired',
        ];

        if ($this->subscriptionModel->update($id, $data)) {
            return redirect()->to('subscriptions')->with('success', 'Subscription disabled successfully');
        }
        return redirect()->to('subscriptions')->with('error', 'Unable to disable Subscription');
    }

    // enable subscriptions
    public function enable($id)
    {
        // dd('enable');
        $data = [
            'status' => 'active',
        ];

        if ($this->subscriptionModel->update($id, $data)) {
            return redirect()->to('subscriptions')->with('success', 'Subscription enabled successfully');
        }
        return redirect()->to('subscriptions')->with('error', 'Unable to disable Subscription');
    }

}
