<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\BSCPerspectivesModel;
use App\Models\GoalsModel;
use App\Models\InitiativesModel;
use App\Models\OrganizationsModel;
use App\Models\ThemesModel;
use CodeIgniter\HTTP\ResponseInterface;

class ThemesController extends BaseController
{
    protected $themesModel;
    protected $bscperspectiveModel;
    protected $goalsModel;
    protected $initiativesModel;
    protected $organizationsModel;
    protected $organization;

    public function __construct()
    {
        $this->themesModel = new ThemesModel();
        $this->bscperspectiveModel = new BSCPerspectivesModel();
        $this->goalsModel = new GoalsModel();
        $this->initiativesModel = new InitiativesModel();
        $this->organizationsModel = new OrganizationsModel();
        $this->organization = $this->organizationsModel->where('slug', session()->get('slug'))->first();
    }

    public function index()
    {
        $status = $this->request->getGet('status');

        $builder = $this->themesModel
            ->where('organization_id', $this->organization['id']);

        if (!is_null($status)) {
            if ($status === 'due') {
                $builder
                    ->where('timeline <', date('Y-m-d H:i:s'))
                    ->where('status !=', 'completed');
            } else {
                // Normal status filter
                $builder->where('status', $status);
            }
        }

        $themes = $builder
            ->orderBy('status')
            ->orderBy('timeline')
            ->findAll();

        $action = $this->request->getGet('action');
        if ($action === 'download') {
            return view('downloads/themes', [
                'themes' => $themes,
                'organization' => $this->organization
            ]);
        }

        return view('themes/index', [
            'organization' => $this->organization,
            'themes' => $themes
        ]);
    }


    public function getThemeData($theme)
    {
        // Step 2: Fetch all goals under this theme, with their BSC perspective
        $goals = $this->goalsModel
            ->select('
            goals.*,
            bscperspectives.name AS perspective_name
        ')
            ->join('bscperspectives', 'bscperspectives.id = goals.bscperspective_id')
            ->where('goals.theme_id', $theme['id'])
            ->orderBy('goals.bscperspective_id')
            ->get()->getResultArray();

        $result = [];

        // Step 3: Loop through each goal and fetch its initiatives + active owners
        foreach ($goals as $goal) {
            $rawInitiatives = $this->initiativesModel
                ->select('
                initiatives.*,
                initiativeowners.id AS owner_link_id,
                initiativeowners.status AS owner_status,
                users.name AS user_owner_name,
                organizationstructures.name AS org_owner_name
            ')
                ->join('initiativeowners', 'initiativeowners.initiative_id = initiatives.id', 'left')
                ->join('users', 'users.id = initiativeowners.individual_owner_id', 'left')
                ->join('organizationstructures', 'organizationstructures.id = initiativeowners.organizationstructure_owner_id', 'left')
                ->where('initiatives.status', 'active')
                ->where('initiatives.goal_id', $goal['id'])
                ->get()->getResultArray();

            $groupedInitiatives = [];

            foreach ($rawInitiatives as $row) {
                $id = $row['id'];

                if (!isset($groupedInitiatives[$id])) {
                    $groupedInitiatives[$id] = $row;
                    $groupedInitiatives[$id]['owners'] = [];
                }

                // Only include owner if status is active
                if ($row['owner_status'] === 'active') {
                    $ownerName = $row['org_owner_name'] ?: $row['user_owner_name'];

                    if ($ownerName && !in_array($ownerName, $groupedInitiatives[$id]['owners'])) {
                        $groupedInitiatives[$id]['owners'][] = $ownerName;
                    }
                }
            }

            $perspectiveName = $goal['perspective_name'];

            // Step 4: Group by perspective
            if (!isset($result[$perspectiveName])) {
                $result[$perspectiveName] = [];
            }

            $result[$perspectiveName][] = [
                'goal' => $goal,
                'initiatives' => array_values($groupedInitiatives)
            ];
        }

        return $result;
    }

    // display the details of a particular theme
    public function view($slug)
    {
        $theme = $this->themesModel->where('slug', $slug)->first();

        $action = $this->request->getGet('action');
        if ($action === 'download') {
            return view('downloads/single_theme', [
                'theme' => $theme,
                'groupedData' => $this->getThemeData($theme),
                'organization' => $this->organization,
            ]);
        }

        return view('themes/view', [
            'theme' => $theme,
            'groupedData' => $this->getThemeData($theme),
            'organization' => $this->organization,
        ]);
    }


    // display form for creating new themes
    public function create()
    {

        return view('themes/create', ['organization' => $this->organization]);
    }

    // collect user data and store in the db
    public function store()
    {
        $rules = [
            'title' => 'required|string',
            'description' => 'required|string',
            'timeline' => 'required',
        ];

        if (!$this->validate($rules)) {
            return view('themes/create', ['validation' => $this->validator, 'organization' => $this->organization]);
        }

        $data = [
            'organization_id' => $this->organization['id'],
            'bscperspective_id' => $this->request->getPost('bscperspective_id') ?: null,
            'slug' => substr(md5(uniqid(time(), true)), 0, 40),
            'title' => $this->request->getPost('title'),
            'description' => $this->request->getPost('description'),
            'timeline' => $this->request->getPost('timeline'),
        ];

        // dd($data);

        if ($this->themesModel->save($data)) {
            return redirect()->to('themes')->with('success', 'Theme created successfully');
        }
    }

    // display form to edit a theme
    public function edit($slug)
    {
        $theme = $this->themesModel->where('slug', $slug)->first();

        return view('themes/edit', ['theme' => $theme, 'organization' => $this->organization]);
    }

    // collect user data and update theme in the db
    public function update()
    {
        $id = $this->request->getPost('id');
        $theme = $this->themesModel->find($id);

        $rules = [
            'title' => 'required|string',
            'description' => 'required|string',
            'timeline' => 'required',
        ];

        if (!$this->validate($rules)) {
            return view('themes/edit', ['validation' => $this->validator, 'organization' => $this->organization]);
        }

        $data = [
            'organization_id' => $this->organization['id'],
            'bscperspective_id' => $this->request->getPost('bscperspective_id') ?: null,
            'title' => $this->request->getPost('title'),
            'description' => $this->request->getPost('description'),
            'timeline' => $this->request->getPost('timeline'),
        ];

        if ($this->themesModel->update($id, $data)) {
            return redirect()->to('themes')->with('success', 'Theme updated successfully');
        }
    }

    // disable theme
    public function disable($id)
    {
        // dd('ok');
        $data = [
            'status' => 'inactive',
        ];

        if ($this->themesModel->update($id, $data)) {
            return redirect()->to('themes')->with('success', 'Theme disabled successfully');
        }
        return redirect()->to('themes')->with('error', 'Unable to disable theme');
    }

    // enable theme
    public function enable($id)
    {
        // dd('enable');
        $data = [
            'status' => 'active',
        ];

        if ($this->themesModel->update($id, $data)) {
            return redirect()->to('themes')->with('success', 'Theme enabled successfully');
        }
        return redirect()->to('themes')->with('success', 'Unable to enable theme');
    }

    // mark a theme complete
    public function markComplete($id)
    {
        $data = [
            'status' => 'completed',
        ];

        if ($this->themesModel->update($id, $data)) {
            return redirect()->to('themes')->with('success', 'Theme marked completed');
        }
    }
}
