<?php

namespace App\Libraries;

use App\Models\EmailsModel;
use App\Models\EmailRecipientsModel;
use Config\Services;

class EmailService
{
    protected $emailsModel;
    protected $emailRecipientsModel;
    protected $email;

    public function __construct()
    {
        $this->emailsModel         = new EmailsModel();
        $this->emailRecipientsModel = new EmailRecipientsModel();
        $this->email               = Services::email();
    }

    /**
     * Process pending email recipients in batches.
     *
     * @param int        $limit           How many recipients to process this run
     * @param int|null   $organizationId  Optional org scope
     * @param bool       $dryRun          If true, don’t send; just list what would be sent
     * @return array{processed:int,sent:int,failed:int,skipped:int,message:string}
     */
    public function processPending(int $limit = 50, ?int $organizationId = null, bool $dryRun = false): array
    {
        // Fetch a small batch of pending recipients with their email content
        $builder = $this->emailRecipientsModel
            ->select('emailrecipients.*, emails.subject, emails.body')
            ->join('emails', 'emails.id = emailrecipients.email_id')
            ->where('emailrecipients.status', 'pending')
            ->where('emails.status', 'pending')
            ->orderBy('emailrecipients.id', 'ASC')
            ->limit($limit);

        if ($organizationId) {
            $builder->where('emailrecipients.organization_id', $organizationId);
        }

        $pendingRecipients = $builder->findAll();

        if (empty($pendingRecipients)) {
            return [
                'processed' => 0, 'sent' => 0, 'failed' => 0, 'skipped' => 0,
                'message' => 'No pending recipients found.'
            ];
        }

        $processedEmailIds = [];
        $sent = $failed = $skipped = 0;

        foreach ($pendingRecipients as $recipient) {
            // Safety checks
            if (empty($recipient['email_address']) || empty($recipient['subject'])) {
                $this->emailRecipientsModel->update($recipient['id'], [
                    'status' => 'failed',
                    'sent_at' => date('Y-m-d H:i:s')
                ]);
                $failed++;
                continue;
            }

            if ($dryRun) {
                // Don’t send, just count; keep status as pending
                $skipped++;
                $processedEmailIds[] = $recipient['email_id'];
                continue;
            }

            // Send email
            $this->email->clear(true);
            $this->email->setTo($recipient['email_address']);
            $this->email->setSubject($recipient['subject']);
            $this->email->setMessage($recipient['body']); // ensure mailType=html in Email config if HTML body

            $ok = $this->email->send();

            if ($ok) {
                $this->emailRecipientsModel->update($recipient['id'], [
                    'status' => 'sent',
                    'sent_at' => date('Y-m-d H:i:s')
                ]);
                $sent++;
                $processedEmailIds[] = $recipient['email_id'];
            } else {
                // Log the failure; keep data for audit
                log_message('error', 'Email send failed to {to}. Debug: {dbg}', [
                    'to' => $recipient['email_address'],
                    'dbg' => $this->email->printDebugger(['headers', 'subject'])
                ]);

                $this->emailRecipientsModel->update($recipient['id'], [
                    'status' => 'failed',
                    'sent_at' => date('Y-m-d H:i:s')
                ]);
                $failed++;
            }
        }

        // Mark parent emails as "sent" if no pending recipients remain
        foreach (array_unique($processedEmailIds) as $emailId) {
            $pendingCount = $this->emailRecipientsModel
                ->where('email_id', $emailId)
                ->where('status', 'pending')
                ->countAllResults();

            if ($pendingCount === 0) {
                $this->emailsModel->update($emailId, ['status' => 'sent']);
            }
        }

        $processed = count($pendingRecipients);
        return [
            'processed' => $processed,
            'sent' => $sent,
            'failed' => $failed,
            'skipped' => $skipped,
            'message' => "Processed {$processed} | sent={$sent}, failed={$failed}, dry-skipped={$skipped}"
        ];
    }
}
