<?php

namespace App\Libraries;

use App\Models\NotificationEmailModel;
use Config\Database;
use Config\Services;

class NotificationEmailService
{
    protected $db;
    protected $notifModel;
    protected $email;

    public function __construct()
    {
        $this->db         = Database::connect();
        $this->notifModel = new NotificationEmailModel();
        $this->email      = Services::email();
    }

    /**
     * Queue notification emails for initiatives & KPIs whose timeline is exactly
     * 29 days, 15 days, or 0 days from $today (inclusive).
     */
    public function queueDue(?string $today = null, array $stages = [29, 15, 0]): array
    {
        $timezone   = new \DateTimeZone('Africa/Lagos');
        $base = $today ? new \DateTimeImmutable($today, $timezone) : new \DateTimeImmutable('now', $timezone);

        // Build the exact target dates (values only)
        $dates = [];
        foreach ($stages as $stage) {
            $dates[$stage] = $base->modify("+{$stage} days")->format('Y-m-d');
        }
        $initiativeDates = array_values($dates); // array of date strings

        $queued = 0;
        $skipped = 0;

        $insertRows = function (array $rows) use (&$queued, &$skipped, $base) {
            if (empty($rows)) return;
            foreach ($rows as $row) {
                // defensive: ensure scalar values
                $row['recipient_email'] = (string) ($row['recipient_email'] ?? '');
                $row['recipient_name']  = (string) ($row['recipient_name'] ?? '');
                $row['type']            = (string) ($row['type'] ?? '');
                $row['name']            = (string) ($row['name'] ?? '');
                $row['timeline']        = (string) ($row['timeline'] ?? null);
                $row['notify_stage']    = isset($row['notify_stage']) ? (int)$row['notify_stage'] : null;

                $exists = $this->notifModel->where([
                    'organization_id' => $row['organization_id'],
                    'recipient_email' => $row['recipient_email'],
                    'recipient_name'  => $row['recipient_name'],
                    'type'            => $row['type'],
                    'name'            => $row['name'],
                    'timeline'        => $row['timeline'],
                    'notify_stage'    => $row['notify_stage'],
                ])->first();

                if ($exists) {
                    $skipped++;
                    continue;
                }

                try {
                    $this->notifModel->insert($row);
                    $queued++;
                } catch (\Throwable $e) {
                    $skipped++;
                    log_message('debug', 'Notification insert failed: ' . $e->getMessage());
                }
            }
        };

        // ================= INITIATIVES =================
        // Note: initiatives use end_date in your DB (as per your code sample)
        $initiatives = $this->db->table('initiatives i')
            ->select('i.id, i.organization_id, i.name, i.end_date')
            ->whereIn('i.end_date', $initiativeDates)
            ->where('i.status', 'active')
            ->get()->getResultArray();

        foreach ($initiatives as $i) {
            $owners = $this->db->table('initiativeowners io')
                ->select('io.individual_owner_id, io.organizationstructure_owner_id AS structure_id')
                ->where('io.initiative_id', $i['id'])
                ->get()->getResultArray();

            $rows = [];
            foreach ($owners as $owner) {
                if (!empty($owner['individual_owner_id'])) {
                    $u = $this->db->table('users')
                        ->select('name,email,status')
                        ->where('id', $owner['individual_owner_id'])
                        ->get()->getRowArray();
                    if ($u && ($u['status'] ?? '') === 'active' && filter_var($u['email'], FILTER_VALIDATE_EMAIL)) {
                        $daysLeft = (int) (new \DateTimeImmutable($i['end_date']))->diff($base)->days;
                        $rows[] = [
                            'organization_id' => $i['organization_id'],
                            'recipient_name'  => $u['name'],
                            'recipient_email' => $u['email'],
                            'type'            => 'initiative',
                            'name'            => $i['name'],
                            'status'          => 'pending',
                            'timeline'        => $i['end_date'],
                            'notify_stage'    => $daysLeft + 1,
                        ];
                    }
                } elseif (!empty($owner['structure_id'])) {
                    $users = $this->db->table('users')
                        ->select('name,email,status')
                        ->where('organizationstructures_id', $owner['structure_id'])
                        ->where('status', 'active')
                        ->get()->getResultArray();

                    foreach ($users as $user) {
                        if (filter_var($user['email'], FILTER_VALIDATE_EMAIL)) {
                            $daysLeft = (int) (new \DateTimeImmutable($i['end_date']))->diff($base)->days;
                            $rows[] = [
                                'organization_id' => $i['organization_id'],
                                'recipient_name'  => $user['name'],
                                'recipient_email' => $user['email'],
                                'type'            => 'initiative',
                                'name'            => $i['name'],
                                'status'          => 'pending',
                                'timeline'        => $i['end_date'],
                                'notify_stage'    => $daysLeft + 1,
                            ];
                        }
                    }
                }
            }
            $insertRows($rows);
        }

        // ================= KPIs =================
        $kpis = $this->db->table('kpis k')
            ->select('k.id, k.organization_id, k.name, k.timeline')
            ->whereIn('k.timeline', $initiativeDates)
            ->get()->getResultArray();

        foreach ($kpis as $kpi) {
            $owners = $this->db->table('kpiowners ko')
                ->select('ko.individual_owner_id, ko.organizationstructure_owner_id AS structure_id')
                ->where('ko.kpi_id', $kpi['id'])
                ->get()->getResultArray();

            $rows = [];
            foreach ($owners as $owner) {
                if (!empty($owner['individual_owner_id'])) {
                    $u = $this->db->table('users')
                        ->select('name,email,status')
                        ->where('id', $owner['individual_owner_id'])
                        ->get()->getRowArray();
                    if ($u && ($u['status'] ?? '') === 'active' && filter_var($u['email'], FILTER_VALIDATE_EMAIL)) {
                        $daysLeft = (int) (new \DateTimeImmutable($kpi['timeline']))->diff($base)->days;
                        $rows[] = [
                            'organization_id' => $kpi['organization_id'],
                            'recipient_name'  => $u['name'],
                            'recipient_email' => $u['email'],
                            'type'            => 'kpi',
                            'name'            => $kpi['name'],
                            'status'          => 'pending',
                            'timeline'        => $kpi['timeline'],
                            'notify_stage'    => $daysLeft + 1,
                        ];
                    }
                } elseif (!empty($owner['structure_id'])) {
                    $users = $this->db->table('users')
                        ->select('name,email,status')
                        ->where('organizationstructures_id', $owner['structure_id'])
                        ->where('status', 'active')
                        ->get()->getResultArray();

                    foreach ($users as $user) {
                        if (filter_var($user['email'], FILTER_VALIDATE_EMAIL)) {
                            $daysLeft = (int) (new \DateTimeImmutable($kpi['timeline']))->diff($base)->days;
                            $rows[] = [
                                'organization_id' => $kpi['organization_id'],
                                'recipient_name'  => $user['name'],
                                'recipient_email' => $user['email'],
                                'type'            => 'kpi',
                                'name'            => $kpi['name'],
                                'status'          => 'pending',
                                'timeline'        => $kpi['timeline'],
                                'notify_stage'    => $daysLeft + 1,
                            ];
                        }
                    }
                }
            }
            $insertRows($rows);
        }

        return ['queued' => $queued, 'skipped' => $skipped];
    }

    // sendPending() stays as you already have it (you just need to personalize message using recipient_name if desired).
}
